import re, subprocess, time

from cnvpnclient.os_utils import run_command
from cnvpnclient.exception import ClientException

# TODO: Refactor in progress: move wireguard specific calls and parsing here.

CONF_REGEX = "wg[0-9]+"

# file permissions for wireguard conf files
CONF_FILE_CHMOD = 0o600

# check for valid cidr
CIDR_CHECK_REGEX = "^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\\.){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])(\\/(\\d|[1-2]\\d|3[0-2]))?$"

"""
Wireguard knowledge: If new to code, please read!

On connections:
- Wireguard claims to be "connection-less". Meaning it doesnt have a notion of connections and therefore
  "connecting" or "disconnecting". For wireguard, an interface either exists (is UP) or does not exist (DOWN).
  Our client effectively adds the concept of a connection (and corresponding states 
  connected, disconnected, connecting, disconnecting).

On interfaces:
- wg-quick accepts paths to conf files OR interface names. The conf file is "more explicit". 
  If interface name is provided, it must be the name of the conf file used AND that conf
  file must be located in wireguards installation directory (e.g. /etc/wireguard/).
- We PREFER to use explicit conf file for wg-quick up/downs BUT if user configured CLI
  to put wg confs in the installation directory (WG_DIR) then interface can be used confidently
  on linux and windows, NOT on mac
- MAC specifics: on mac, the interface created will NOT correspond to the name of the conf file.
  Instead, it will be given the "next" utun interface available. (even if conf file is named utunX.conf. So
  you could have a mac conf file named utun5.conf and have that correspond to interface utun7 for example).
"""


class WGErrors(object):
    INVALID_WG_PATH = "Invalid wireguard path"


def windows_dump_log(env, tail=False):
    cmd = "%s /dumplog" % env.get("wg_quick_path")
    if tail:
        cmd = "%s /tail | select" % cmd
    else:
        cmd = "%s | select" % cmd
    subprocess.call(cmd)


def get_version(env, wg_path):
    version_cmd = "%s --version" % wg_path
    if env.is_windows:
        command = ["powershell", version_cmd]
    else:
        command = version_cmd

    version_resp_std, _ = run_command(command, windows=env.is_windows)
    if not version_resp_std:
        return (None, "%s: %s" % (WGErrors.INVALID_WG_PATH, wg_path))
    else:
        version_match = re.search(r"v([0-9\.]+)", version_resp_std)
        if not version_match:
            return (None, "Could not determine wireguard version. Path=%s" % wg_path)
        else:
            return (version_match[1], None)


def wg_quick_test(env, wg_quick_path):
    if env.is_windows:
        _, stderr = run_command(["powershell", "%s /help | select" % wg_quick_path], windows=True)
        has_valid_path = "installtunnelservice" in stderr.lower()
    else:
        _, stderr = run_command("%s" % wg_quick_path)
        has_valid_path = stderr.lower().startswith("usage")

    if not has_valid_path:
        return "wg-quick call failed. Path=%s. Stderr=%s" % (wg_quick_path, stderr)
    return None


def verify_wg(env):
    version, wg_error = get_version(env, env.get("wg_path"))
    wg_quick_error = wg_quick_test(env, env.get("wg_quick_path"))

    errors = {}
    if wg_error:
        errors["wg"] = wg_error
    if wg_quick_error:
        errors["wg-quick"] = wg_quick_error

    is_valid = not (wg_error or wg_quick_error)

    return is_valid, version, errors


def parse_show_output(output):
    lines = output.split("\n")
    data = {}
    peers = []
    _cur_peer = None
    for l in lines:
        line_cln = l.strip()
        if not line_cln:
            if _cur_peer is not None:
                peers.append(_cur_peer)
            _cur_peer = None
            continue

        line_parts = line_cln.split(":")
        key = line_parts[0].strip().lower().replace(" ", "_")
        val = line_parts[1].strip()
        if key == "peer":
            _cur_peer = {"public_key": val}
        elif _cur_peer is not None:
            _cur_peer[key] = val
        else:
            data[key] = val

    if _cur_peer is not None:
        peers.append(_cur_peer)

    data["peers"] = {peer["public_key"]: peer for peer in peers}
    return data


def parse_show_error(error):
    stderr_low = error.lower()
    if "permission denied" in stderr_low:
        return "Interface exists but invalid permissions. Please run as admin for full details"
    elif "unable to access interface" in stderr_low:
        return "Interface does not exist"
    else:
        #return WGErrors.INVALID_WG_PATH
        return error


def get_interface_details(env, wg_path, interface, retry_attempts=0, retry_sleep=1.0):
    retry_attempts = retry_attempts or 0 # handle falsy like none
    cmd = "%s show %s" % (wg_path, interface)
    total_attempts = retry_attempts + 1
    last_error = None
    for i in range(total_attempts):
        if env.is_windows:
            stdout, stderr = run_command(["powershell", cmd])
        else:
            stdout, stderr = run_command(cmd)
        if not stdout or "latest handshake" not in stdout:
            last_error = parse_show_error(stderr)
            can_retry = "does not exist" in last_error or "latest handshake" not in stdout # only "non-fatal" error
            if not can_retry or i == retry_attempts:
                return None, "%s [interface=%s] [wg=%s]" % (last_error, interface, wg_path)
        else:
            return parse_show_output(stdout), None
        time.sleep(retry_sleep)


def is_interface_down(env, interface):
    _show_cmd = "%s show %s" % (env.get("wg_path"), interface)
    if env.is_windows:
        command = ["powershell", _show_cmd]
    else:
        command = _show_cmd

    out, err = run_command(command, windows=env.is_windows)
    if err and "permission" in err and env.logger:
        env.logger.error(
            "Invalid permissions for wireguard show interface. Please run as admin."
        )
    return out == "" or not interface in out


def is_interface_up(env, interface):
    return not is_interface_down(env, interface)


def handle_wg_quick_output(stdout, err, logger=None):
    if err and "error" in err.lower():
        logger.info(f"wg-quick error: {err}")
        raise ClientException(err)
    return (stdout, None)


def build_quick_env(env):
    quick_vars = [
        "WG_PATH=%s" % env.get("wg_path"),
    ]
    wireguard_impl = env.get("wg_impl_binary")
    if wireguard_impl:
        quick_vars.append("WG_QUICK_USERSPACE_IMPLEMENTATION=%s" % wireguard_impl)
    return " ".join(quick_vars)


def wg_quick_up(env, interface=None, conf_file_path=None):
    """wg_quick_up

    Run wg-quick up. wg-quick will accept either interface name or path to a conf file. If
    an interface is passed and we see that it is already down, we return early. the conf
    file will always take precedence over interface in the command as it is more "explicit"

    Args:
        env (config.EnvConfig):
        interface (str, optional): interface name
        conf_file_path (str, optional): path to conf file

    Returns:
        (tuple): (stdout, stderr)
    """
    conf_file_path = conf_file_path or interface
    if is_interface_up(env, interface):
        return (None, None)

    if env.is_windows:
        return run_command(
            [
                "powershell",
                f"{env.get('wg_quick_path')} /installtunnelservice '{conf_file_path}'",
            ],
            windows=True
        )
    else:
        stdout, err = run_command(
            f"{build_quick_env(env)} {env.get('wg_quick_path')} up '{conf_file_path}'",
            logger=env.logger,
        )
        return handle_wg_quick_output(stdout, err, logger=env.logger)


def wg_quick_down(env, interface=None, conf_file_path=None):
    """wg_quick_down

    Run wg-quick down. wg-quick will accept either interface name or path to a conf file. If
    an interface is passed and we see that it is already down, we return early. the conf
    file will always take precedence over interface in the command as it is more "explicit"

    Args:
        env (config.EnvConfig):
        interface (str, optional): interface name
        conf_file_path (str, optional): path to conf file

    Returns:
        (tuple): (stdout, stderr)
    """
    conf_file_path = conf_file_path or interface

    if interface and is_interface_down(env, interface):
        return (None, None)

    if env.is_windows:
        if not interface:
            if env.logger:
                env.logger.warning(
                    "wg-quick down called on windows but no interface provided"
                )
            return (None, None)

        return run_command(
            [
                "powershell",
                f"{env.get('wg_quick_path')} /uninstalltunnelservice {interface}",
            ],
            windows=True
        )
    else:
        stdout, err = run_command(
            f"{build_quick_env(env)} {env.get('wg_quick_path')} down {conf_file_path}",
            logger=env.logger,
        )
        return handle_wg_quick_output(stdout, err, logger=env.logger)
