import os
import re
import shutil
import signal
import subprocess
import time
import socket

try:
    import pwd
except ModuleNotFoundError:
    pass

from cnvpnclient.exception import ClientException

# utility method to run a system command
def run_command(command, logger=None, should_raise=False, windows=False, **kwargs):
    process_kwargs = {
        "stdout": subprocess.PIPE,
        "stderr": subprocess.PIPE,
        "shell": True,
    }

    if windows:
        process_kwargs.update(creationflags=subprocess.CREATE_NO_WINDOW)

    if kwargs:
        process_kwargs.update(kwargs)

    with subprocess.Popen(command, **process_kwargs) as proc:
        output, error = proc.communicate()
        if output is not None:
            output = output.decode("utf-8").strip()
        if error is not None:
            error = error.decode("utf-8").strip()

        if error:
            error_str = "Command stderr. command=%s: %s" % (command, error)
            if logger:
                logger.debug(error_str)

            proc.terminate()
            if should_raise:
                raise ClientException(error_str)
            return (output, error)
        return (output, error)


def process_exists_win(pid):
    try:
        stdout, _ = run_command(["powershell", "Get-Process -Id %s" % pid], windows=True)
        return stdout != ""
    except Exception as e:
        return False


def process_exists_posix(pid):
    try:
        os.kill(int(pid), 0)
    except (ValueError, ProcessLookupError, OSError, SystemError) as e:
        if "not permitted" in str(e).lower():
            # None means, i dont know
            return None
        return False
    else:
        return True


def process_exists(env, pid):
    """Check For the existence of a unix pid."""
    if not all([pid, pid.isdigit()]):
        return False

    if env.is_windows:
        return process_exists_win(pid)
    # posix
    return process_exists_posix(pid)


def wait_for_process_killed(env, pid, max_tries=5, sleep_secs=0.5):
    tries = 0
    while tries < max_tries:
        _exists = process_exists(env, pid)
        if _exists is False:
            return True
        elif _exists is None:
            raise ClientException("Permission error: cannot determine process state")
        tries = tries + 1
        time.sleep(sleep_secs)
    return False


def kill_process(pid, code=signal.SIGTERM):
    try:
        os.kill(int(pid), code)
        return True
    except (ValueError, ProcessLookupError, OSError, SystemError) as e:
        if "not permitted" in str(e).lower():
            # None means, i dont know. But probably not.
            return None
        return False


def find_windows_parent_process(pid):
    cmd = [
        "powershell",
        "Get-WmiObject Win32_Process -Filter \"ProcessId = '%s'\"" % pid,
    ]

    out, err = run_command(cmd, windows=True)
    if err:
        return None

    parent_match = re.search(r"ParentProcessId[\ ]+\:[\ ]*([0-9]+)", out)
    if parent_match:
        parent_pid = parent_match[1]
        if parent_pid.isdigit():
            return int(parent_pid)
    return None


def kill_process_win(pid):
    # parent_pid = find_windows_parent_process(pid)
    try:
        return kill_process(pid, code=signal.SIGTERM)
        # if parent_pid:
        # return kill_process(pid, code=signal.CTRL_BREAK_EVENT)
        # else:
        # return kill_process(pid, code=signal.CTRL_BREAK_EVENT)
    except (ValueError, ProcessLookupError, OSError):
        return False


def kill_env_process(env, pid, force=False):
    if env.is_windows:
        return kill_process_win(pid)
    else:
        signal_code = signal.SIGKILL if force else signal.SIGTERM
        return kill_process(pid, signal_code)


def ensure_owner_posix(file_path, user, group=None, allow_fail=False):
    if not os.path.exists(file_path):
        return False

    if type(user) is int or user.isdigit():
        user = pwd.getpwuid(int(user)).pw_name

    kwargs = {"user": user}
    if group:
        kwargs.update(group=group)
    try:
        shutil.chown(file_path, **kwargs)
    except PermissionError as e:
        if allow_fail:
            return False
        raise e

    return True


def find_ip_interface(env, ip, retries=0, sleep=0.5):
    if env.is_macos:
        # Note: assuming ifconfig in sbin in all distros for macos
        count_str, stderr = run_command("ifconfig | grep %s | wc -l" % ip)
        if stderr:
            raise ClientException("Failed to find ip interface. Error:%s" % stderr)

        if not (count_str and count_str.isdigit() and int(count_str) == 1):
            if retries > 0:
                time.sleep(sleep)
                return find_ip_interface(env, ip, retries=(retries - 1), sleep=sleep)
            raise ClientException(
                "Could not determine interface for IP. Found %s interfaces." % count_str
            )

        cmd = "ifconfig | grep -B 1 %s" % ip
        out, error = run_command(cmd)
        if error:
            raise ClientException("ifconfig not supported.")
        if out:
            return out.split(":")[0]
    else:
        raise ClientException("find ip by interface not supported!")
    return None


def reverse_readline(filename, buf_size=8192, max_memory=None):
    """reverse_readline

    A generator that returns the lines of a file in reverse order.
    (stack overflow steal)

    Args:
        filename (str):
        buf_size (int, optional):  Defaults to 8192.

    Yields: str
    """
    with open(filename) as fh:
        segment = None
        offset = 0
        fh.seek(0, os.SEEK_END)
        file_size = remaining_size = fh.tell()
        max_memory = max_memory or (file_size + 100)
        while remaining_size > 0 and (file_size - remaining_size < max_memory):
            offset = min(file_size, offset + buf_size)
            fh.seek(file_size - offset)
            buffer = fh.read(min(remaining_size, buf_size))
            remaining_size -= buf_size
            lines = buffer.split("\n")
            # The first line of the buffer is probably not a complete line so
            # we'll save it and append it to the last line of the next buffer
            # we read
            if segment is not None:
                # If the previous chunk starts right from the beginning of line
                # do not concat the segment to the last line of new chunk.
                # Instead, yield the segment first
                if buffer[-1] != "\n":
                    lines[-1] += segment
                else:
                    yield segment
            segment = lines[0]
            for index in range(len(lines) - 1, 0, -1):
                if lines[index]:
                    yield lines[index]
        # Don't yield None if the file was empty
        if segment is not None:
            yield segment


# resolve host to ip, keep the host to ip list for when the connection goes down
# the host to ip can still be resolved
resolved_host_ips = {}


def resolve_ip(host):
    try:
        ip = resolved_host_ips[host] = socket.gethostbyname(host)
        return ip
    except Exception as e:
        if host in resolved_host_ips:
            return resolved_host_ips[host]
        else:
            raise e
