import logging
import os
import json

LOG_FORMAT = "[%(asctime)s] [name=%(name)s] [level=%(levelname)s] %(message)s"


def json_stringify(val):
    return json.dumps(val, sort_keys=True, default=str)


def clear_handlers(logger):
    while logger.hasHandlers():
        logger.removeHandler(logger.handlers[0])


def setup_root_logging_stdout(log_level=None):
    import sys

    logger = logging.getLogger("cnvpnclient")
    if log_level is not None:
        _log_level = log_level.upper() if type(log_level) is str else log_level
        if _log_level not in ["DEBUG", "INFO", "WARNING", "ERROR"]:
            _log_level = "INFO"
    else:
        _log_level = os.getenv("CNVPN_LOG_LEVEL", "INFO").upper()

    logger.setLevel(_log_level)
    handler = logging.StreamHandler(sys.stdout)
    handler.setLevel(_log_level)
    formatter = logging.Formatter(LOG_FORMAT)
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    return logger


def configure_client_logger(name, log_level, log_file, log_format):
    """configure_client_logger

    Args:
        name (str): name of connection
        log_level (int or str): valid log level to set
        log_file (str): path to log file
        log_format (str): format string

    Returns: logger object
    """
    if not all([name, log_level, log_file, log_format]):
        return None
    logger = logging.getLogger(name)
    clear_handlers(logger)
    logger.setLevel(log_level)
    log_handler = logging.handlers.RotatingFileHandler(
        log_file, mode="a", maxBytes=(5 * 1024 * 1024), backupCount=3
    )
    log_handler.setLevel(log_level)
    formatter = logging.Formatter(log_format)
    log_handler.setFormatter(formatter)
    logger.addHandler(log_handler)
    return logger
