import os
import re
import sys
import getpass

# import ctypes
import platform
from pathlib import Path


def read_env_platform():
    return {"system": platform.system().lower()}


def get_username():
    try:
        return str(os.getlogin())
    except OSError:
        return getpass.getuser()


def clean_connection_name(name):
    # only allow alphanumeric naming
    return re.sub("[^0-9a-zA-Z_]+", "_", name).strip("_")


class ConnectionTypes:
    WIREGUARD_NATIVE = "wg-native"
    WIREGUARD_COHESIVE = "wg-cohesive"
    # OPENVPN = 'openvpn' # not supported yet
    # future: will have to either 1) make type required (no default) or 2) parse
    # "type" from file format.
    DEFAULT = WIREGUARD_COHESIVE
    ALL = [WIREGUARD_NATIVE, WIREGUARD_COHESIVE]


class LibConfig:
    LIB_DIR = Path(__file__).resolve().parent
    PROJ_DIR = LIB_DIR.parent
    BIN_DIR = os.path.join(LIB_DIR, "bin")
    CLI_PATH = os.path.join(BIN_DIR, "cnvpn")
    DEFAULT_USER_DATA_HOME_PATH = (".cohesive",)
    DEFAULT_WIN_USER_DATA_HOME_PATH = ("AppData", "Local", "Cohesive")
    CONN_DATA_FILE_PREFIX = "connection"
    CONN_ROUTES_FILE_PREFIX = "routes"
    SUBPROCESS = "subprocess"
    THREAD = "thread"
    DATETIME_FORMAT = "%Y-%m-%d %H:%M:%S"
    SUPPORTED_CLIENT_RUNTIMES = [SUBPROCESS, THREAD]
    DEFAULT_CONNECT_TIMEOUT = 30

    MIN_POLLING_INTERVAL = 30
    MIN_HANDSHAKE_TIMEOUT = 10
    # defaults
    class Defaults:
        POLLING_INTERVAL = 30
        HANDSHAKE_TIMEOUT = 180
        ROUTE_POLLING = False
        TUNNEL_ALL_TRAFFIC = False

    @classmethod
    def default_data_dir(cls, env):
        # use same for windows?
        if env.is_windows:
            # windows app data dir
            data_dir_path = cls.DEFAULT_WIN_USER_DATA_HOME_PATH
        else:
            # mac and linux store in dot directory of user
            data_dir_path = cls.DEFAULT_USER_DATA_HOME_PATH

        return os.path.join(env.user_home, *data_dir_path)

    @classmethod
    def logs_dir(cls, data_dir):
        return os.path.join(data_dir, "logs")

    @classmethod
    def wg_confs_dir(cls, data_dir):
        return os.path.join(data_dir, "confs", "wg")

    @classmethod
    def to_connection_data_file(cls, data_dir, name):
        # other cleaning for name?
        conn_file = "%s.%s" % (cls.CONN_DATA_FILE_PREFIX, clean_connection_name(name))
        return os.path.join(data_dir, conn_file)

    @classmethod
    def to_connection_routes_file(cls, data_dir, name):
        # other cleaning for name?
        conn_file = "%s.%s" % (cls.CONN_ROUTES_FILE_PREFIX, clean_connection_name(name))
        return os.path.join(data_dir, conn_file)

    @classmethod
    def to_connection_log_file(cls, data_dir, name):
        # TODO: rotation? what if it gets too big
        log_file = "%s.%s.log" % (
            cls.CONN_DATA_FILE_PREFIX,
            clean_connection_name(name),
        )
        return os.path.join(cls.logs_dir(data_dir), log_file)

    @classmethod
    def ensure_dirs(cls, data_dir):
        # ensure data dir exists
        dirs = [
            data_dir,
            cls.logs_dir(data_dir),
            cls.wg_confs_dir(data_dir),  # might not be used if WG_DIR set
        ]

        for directory in dirs:
            if not os.path.exists(directory):
                os.makedirs(directory)


class EnvConfig:
    ENV_VAR_PREFIX = "CNVPN"
    # TODO: support openvpn
    ConfigDefaults = {
        "WG_AUTH": "oidc",
        "POLLING_INTERVAL": 30,
        "HANDSHAKE_TIMEOUT": 180,
        "DATA_DIR": None,
        "WG_DIR": None,
        "CLI_PATH": LibConfig.CLI_PATH,
        "CLI_PROXY_ARGS": None,
        "CONN_MONITOR_INTERVAL": 30,
        "LOGIN_RETRY_INTERVAL": 5,
        "AUTO_DISCOVER_INTERFACE": True,
        "ALLOW_CUSTOM_INTERFACE": False,
        "LOG_LEVEL": "INFO",
        "WG_PATH": None,  # default assume in path
        "WG_QUICK_PATH": None,  # default assume in path
        "LOG_FORMAT": "[%(asctime)s] [level=%(levelname)s] %(message)s",
    }

    DefaultWG = "wg"
    DefaultWGQuick = "wg-quick"
    DefaultWGQuickDarwin = os.path.join(
        LibConfig.PROJ_DIR, "scripts", "wg-quick.darwin.bash"
    )

    LOG_LEVELS = ["DEBUG", "INFO", "WARNING", "ERROR"]

    int_types = [
        "POLLING_INTERVAL",
        "LOGIN_RETRY_INTERVAL",
        "CONN_MONITOR_INTERVAL",
    ]
    bool_types = ["AUTO_DISCOVER_INTERFACE", "USE_FILENAME_INTERFACE"]
    # Windows specifics
    WIN_WIREGUARD_LOC1 = "C:\Program Files\WireGuard"
    WIN_WIREGUARD_LOC2 = "C:\Program Files (x86)\WireGuard"

    def __init__(self, **config):
        self._platform = read_env_platform()
        self.config = self.ConfigDefaults.copy()
        self.update_from_env(config)
        runtime_config = self.determine_runtime_config()
        for key, val in runtime_config.items():
            self._set(key, val)

        # cached properties
        self._username = None
        self._user_home = None

        data_dir = self.get("data_dir")
        if not data_dir:
            self._set("data_dir", LibConfig.default_data_dir(self))

        # set WG conf location to data dir if local or as default
        wg_dir = self.get("wg_dir")
        if not wg_dir or wg_dir == "local":
            self._set("wg_dir", LibConfig.wg_confs_dir(self.get("data_dir")))
        LibConfig.ensure_dirs(self.get("data_dir"))

        self.data_dir = self.get("data_dir")
        self.logger = None
        self.state_file = None
        self.is_admin = self.get_is_user_admin()
        _log_level = self.get("log_level")
        if _log_level.upper() not in self.LOG_LEVELS:
            self._set("log_level", self.ConfigDefaults["LOG_LEVEL"])

    def update_from_env(self, config_overrides=None):
        config_overrides = config_overrides or {}
        for key in self.ConfigDefaults.keys():
            override_val = config_overrides.get(key)
            if override_val:
                config_val = override_val
            else:
                config_val = os.getenv("%s_%s" % (self.ENV_VAR_PREFIX, key))

            if config_val:
                if key in self.int_types:
                    val_typed = int(config_val)
                elif key in self.bool_types:
                    val_typed = config_val.lower() not in ["false", "0"]
                else:
                    val_typed = config_val
                self.config[key] = val_typed

    def get(self, name: str):
        return self.config.get(name.upper()) or os.getenv(name.upper())

    def set(self, name, val):
        return self._set(name, val)

    def _set(self, name, val):
        name_up = name.upper()
        self.config[name_up] = val
        return self.config

    def conn_data_file(self, name):
        return LibConfig.to_connection_data_file(self.get("data_dir"), name)

    def conn_routes_file(self, name):
        return LibConfig.to_connection_routes_file(self.get("data_dir"), name)

    def conn_log_file(self, name):
        return LibConfig.to_connection_log_file(self.get("data_dir"), name)

    def determine_runtime_config(self):
        is_frozen = getattr(sys, "frozen", False)
        _config_updates = {}
        if is_frozen:
            _config_updates["py_path"] = None
            _config_updates["cli_path"] = sys.executable
            _config_updates.update(self.read_bundle_data())
        else:
            _config_updates["py_path"] = sys.executable
            _config_updates.update(self.read_build_data())

        has_bundled_wg = "wg_path" in _config_updates
        has_env_wg = self.get("wg_path") is not None
        has_env_wg_quick = self.get("wg_quick_path") is not None

        if self.is_windows:
            if has_bundled_wg:
                _config_updates["wg_path"] = '& "%s"' % _config_updates["wg_path"]
                _config_updates["wg_quick_path"] = (
                    '& "%s"' % _config_updates["wg_quick_path"]
                )
            else:
                if _config_updates["py_path"]:
                    _config_updates["py_path"] = '& "%s"' % _config_updates["py_path"]

                win_wg_exes = self.find_windows_wg_exes()
                if win_wg_exes:
                    if not has_env_wg:
                        _config_updates["wg_path"] = win_wg_exes["wg_path"]
                    if not has_env_wg_quick:
                        _config_updates["wg_quick_path"] = win_wg_exes["wg_quick_path"]
        elif self.is_macos and not has_bundled_wg:
            if not has_env_wg:
                _config_updates["wg_path"] = self.DefaultWG
            if not has_env_wg_quick:
                _config_updates["wg_quick_path"] = self.DefaultWGQuick
        elif not has_bundled_wg:
            if not has_env_wg:
                _config_updates["wg_path"] = self.DefaultWG
            if not has_env_wg_quick:
                _config_updates["wg_quick_path"] = self.DefaultWGQuick
        return _config_updates

    def _read_data_file(self, file_path):
        if not os.path.exists(file_path):
            return {}

        data = {}
        with open(file_path, "r") as bundle_file:
            for line in bundle_file.readlines():
                parts = line.strip().split(":")
                if len(parts) > 1:
                    data[parts[0].lower()] = ":".join(parts[1:]).lower()
        return data

    def read_build_data(self):
        return self._read_data_file(os.path.join(LibConfig.LIB_DIR, "BUILDNUMBER"))

    def read_bundle_data(self):
        bundle_data = self._read_data_file(os.path.join(LibConfig.PROJ_DIR, "BUNDLE"))
        if not bundle_data:
            return {}

        config_updates = {
            "bundled": True,
            "python_version": sys.version,
        }

        wg_bundled = bundle_data.get("wg", None)
        wg_quick_impl = None
        if wg_bundled and wg_bundled.lower() != "false":
            if self.is_windows:
                wg_exe = "wg.exe"
                wg_quick_exe = "wireguard.exe"
            elif self.is_macos:
                wg_exe = "wg"
                wg_quick_exe = "wg-quick.darwin.bash"
                wg_quick_impl = "wireguard-go"
            else:
                wg_exe = "wg"
                wg_quick_exe = "wg-quick"

            wg_path_parts = ("wireguard", "bin", wg_exe)
            wg_quick_path_parts = ("wireguard", "bin", wg_quick_exe)

            config_updates["wg_path"] = os.path.join(LibConfig.PROJ_DIR, *wg_path_parts)
            config_updates["wg_quick_path"] = os.path.join(
                LibConfig.PROJ_DIR, *wg_quick_path_parts
            )
            if wg_quick_impl:
                config_updates["wg_impl_binary"] = os.path.join(
                    LibConfig.PROJ_DIR, "wireguard", "bin", wg_quick_impl
                )

        if "build" in bundle_data:
            config_updates["build"] = bundle_data["build"].strip()
        return config_updates

    @property
    def platform(self):
        return self._platform["system"]

    @property
    def is_linux(self):
        return self.platform == "linux"

    @property
    def is_macos(self):
        return self.platform == "darwin"

    @property
    def is_windows(self):
        return self.platform == "windows"

    @property
    def is_posix(self):
        return not self.is_windows

    @property
    def user_home(self):
        if not self._user_home:
            self._user_home = str(Path.home())
        return self._user_home

    @property
    def username(self):
        if not self._username:
            self._username = get_username()

        return self._username

    def get_is_user_admin(self):
        try:
            is_admin = os.getuid() == 0
        except AttributeError:
            import ctypes

            is_admin = ctypes.windll.shell32.IsUserAnAdmin() != 0
        return is_admin

    def find_windows_wg_exes(self):
        install_dir = None
        if os.path.exists(self.WIN_WIREGUARD_LOC1):
            install_dir = self.WIN_WIREGUARD_LOC1
        elif os.path.exists(self.WIN_WIREGUARD_LOC2):
            install_dir = self.WIN_WIREGUARD_LOC2

        if install_dir:
            return {
                "wg_path": "& '%s'" % os.path.join(install_dir, "wg.exe"),
                "wg_quick_path": "& '%s'" % os.path.join(install_dir, "wireguard.exe"),
            }
        return {}


def read_env(**kwargs):
    return EnvConfig(**kwargs)
