import threading
import requests, os, subprocess, time, logging, struct, signal, re
import socket, ipaddress, urllib3, binascii, json, atexit, sys, traceback
from datetime import datetime, timedelta
from functools import partial
from urllib3.exceptions import InsecureRequestWarning
from os import path
from requests.auth import HTTPBasicAuth
from pathlib import Path
import webbrowser
import logging.handlers

import cnvpnclient.files as file_utils
from cnvpnclient import os_utils, wireguard
from cnvpnclient.exception import ClientException
from cnvpnclient.logutil import json_stringify, configure_client_logger
from cnvpnclient.config import LibConfig, ConnectionTypes
from cnvpnclient.routing_agent import RoutingAgent

lib_logger = logging.getLogger("cnvpnclient")

requests.packages.urllib3.disable_warnings(category=InsecureRequestWarning)

"""
Documentation on props structure
props {
  # These props are parsed or calculated from the clientpack directly. cn=cohesive specific
  Endpoint: str, endpoint for vns3 controller to connect to 
  TunnelAllTraffic: bool, add route for 0.0.0.0/0 to send all internet traffic through vpn [cn]
  SecondaryEndpoints: str, csv of secondary endpoints for failover [cn]
  RoutePolling: bool, if true, will poll VNS3 for routes [cn]
  PresharedKey: str, wg property
  ActiveInterface: str, actual interface up on OS. different based on OS.
  PublicKey: str, wg property
  PrivateKey: str, wg property
  Secret: str, API token to use for sending requests to VNS3 [cn]
  ClientpackId: str, VNS3 clientpack ID [cn]
  Address: str, wg property
  AllowedIPs: str, csv, wg property
  LocalRoutes: str, csv - local routes to add for VPN overlay network
  RoutePollingInterval: int, time (seconds) to wait between polling routes
  HandshakeTimeout: int, time (seconds) that defines a timeout for the connection
  ConnectionStatus: str, store state of connection. empty string or 'up"
  
  overlay_ip: str, this is the Address of the conf and is this clients IP
  interface: str, interface requested (not necessarily the ActiveInterface)
  wg_conf_file: str, path to wireguard conf file
  wg_auth: str, type of auth, not sure we use this anymore
  controller_vip: str, fetched from VNS3 when connecting
  openid: str, state string indicating if openid auth configured on server
  token: str, retrieved after openid login
  refresh_token: str, retrieved after openid login
  routes_file: str, path to file to store routes in
  type: str, ConnectionTypes. wg-native wg-cohesive
  name: str, name of connection
  state_file: str, path to state file
  _auth_attempts: int, state tracker for openid auth attempts
}
"""


def get_current_endpoint(props):
    _endpoint = props.get("Endpoint")
    if not _endpoint:
        return None
    parts = _endpoint.rsplit(":", 1)
    return parts[0].strip()

def get_current_vip(props, logger):
    logger = logger or lib_logger
    if props["ControllerVIP"] == "":
        endpoint = get_current_endpoint(props)
        return endpoint
    _endpoint = props.get("ControllerVIP")
    return _endpoint.split(":")[0]

def update_props_endpoint(props, new_endpoint):
    endpoint_parts = props["Endpoint"].rsplit(":", 1)
    port = endpoint_parts[1] if len(endpoint_parts) > 1 else ""
    new_endpoint = f"{new_endpoint}" if ":" in new_endpoint else new_endpoint
    update = "%s:%s" % (new_endpoint, port) if port else new_endpoint
    props.update(Endpoint=update)
    # mutates and returns
    return props

def get_secondary_endpoints_props(props, allow_duplicates=False):
    """get_secondary_endpoints_props

    Parse and clean SecondaryEndpoints from props dict

    Args:
        props (dict)

    Returns: list[str]
    """
    secondary_endpoints = props.get("SecondaryEndpoints", "")
    current_endpoint = get_current_endpoint(props)
    if current_endpoint not in secondary_endpoints:
        secondary_endpoints = current_endpoint + "," + secondary_endpoints
    endpoints_cln = [x.strip() for x in secondary_endpoints.split(",")]
    # filter empty strings
    endpoints_cln = [endpoint for endpoint in endpoints_cln if endpoint]
    if allow_duplicates:
        return endpoints_cln
    return list(dict.fromkeys(endpoints_cln))

def get_endpoint_retry_order(props):
    secondary_endpoints = get_secondary_endpoints_props(props, True)
    return secondary_endpoints


def find_hosts_file(env):
   if env.is_linux or env.is_macos:
      file_location = "/etc/hosts"
      if path.exists(file_location):
         return file_location

   if env.is_windows:
      drive = os.environ['SYSTEMDRIVE']
      file_locations = [
                      fr"{drive}\windows\system32\drivers\etc\hosts", #Windows XP Home\Vista\Windows 7\8\10
                      fr"{drive}\winnt\system32\drivers\etc\hosts", #Windows NT\2000\XP Pro
                      fr"drivers\etc\hosts{drive}\windows\hosts" #Windows 95\98\Me
                    ]
      for file_location in file_locations:
         if path.exists(file_location):
            return file_location

   raise ClientException(
        f"Could not find hosts file, this is needed for OIDC authentication. Please contact support."
         )

def add_hosts_entry(env, ip, hostname):
    if ip is None or hostname is None:
        return
    hosts_file = find_hosts_file(env)
    entry = f"{ip}  {hostname}"
    with open(hosts_file, "r+") as file:
      lines = file.readlines()
      for line in lines:
        if ip in line:
           return
      file.write(f"{entry}\n")
    file.close()


def add_endpoint_to_host_file(env, props):
    if "ControllerVIP" in props and "OpenIdCallbackHost" in props:
        add_hosts_entry(env, props['ControllerVIP'], props['OpenIdCallbackHost'])

def remove_hosts_entry(env, ip, hostname):
    if ip is None or hostname is None:
       return
    hosts_file = find_hosts_file(env)
    new_contents = ''
    entry = f"{ip}  {hostname}"
    save_file = False
    with open(hosts_file, "r") as file:
      for line in file.readlines():
        if ip not in line:
          new_contents += line
        else:
          save_file = True
    file.close()
    #only write file if necessary
    if save_file and len(new_contents) != 0:
      with open(hosts_file, "w") as file:
        file.write(new_contents)
      file.close()

def remove_endpoint_from_host_file(env, props):
    if "ControllerVIP" in props and "OpenIdCallbackHost" in props:
        remove_hosts_entry(env, props['ControllerVIP'], props['OpenIdCallbackHost'])


# we can verify this based on the login request to controller
def check_login_response(props, response, logger):
    logger = logger or lib_logger
    if 'error' in response and 'oidc_redirect_host' in response['error']:
        props["OpenIdCallbackHost"] = response['error']['oidc_redirect_host']
        #props["NoVIPIncluded"] = False
    if 'error' in response and 'oidc_provider' in response['error']:
        props["OpenIdProviderHost"] = response['error']['oidc_provider']
        props["OpenIdProviderIp"] = response['error']['oidc_provider_ip']

def send_notification(env, message):
    try:
        message = f"\n\n\033[1;30m{message}\033[0m\n\n"
        terminal = subprocess.run(['tty'], capture_output=True, text=True).stdout.strip()
        terminal_fd = os.open(terminal, os.O_WRONLY)
        os.write(terminal_fd, message.encode())
    except Exception as e:
        env.logger.info(f"Could not send message to terminal: {message} %s" % str(e))

def browser_not_supported(props):
    return 'browser_supported' in props and not props['browser_supported']
    
# fetch the connection info for the clientpack
# thats needed to make the wireguard connection
def wg_connect(env, props, clientpack_id, api_token, logger=None, timeout=None):
    """wg_connect

        fetch the connection info for the clientpack
        thats needed to make the wireguard connection
    Args:
        props (dict): client properties
        clientpack_id (str): id/name of clientpack
        api_token (str): API token
        logger (optional): Defaults to library logger.

    Returns: 200 indicating success or None
    """
    logger = logger or lib_logger
    if no_vip_included(props):
       controller = get_current_endpoint(props)
    else:
       controller = get_current_vip(props, logger)
    my_headers = {
        "Content-Type": "application/json",
        "api-token": api_token,
        "state": props["openid"],
        "token": props["token"],
        "refresh-token": props["refresh_token"],
    }
    logger.debug(f"Connecting to controller: {get_current_endpoint(props)} for clientpack {clientpack_id} via {controller}")
    try:
        response = requests.get(
            f"https://{controller}:8000/api/wg-connection/{clientpack_id}",
            headers=my_headers,
            verify=False,
            timeout=(timeout or LibConfig.DEFAULT_CONNECT_TIMEOUT),
        )
    except Exception as e:
        logger.debug("Failed to access controller via %s" % str(e))
        return None

    if "DOCTYPE".lower() in response.text.lower() or "message\":\"VPNCubed::API::UnauthorizedError" in response.text:
        props["_auth_attempts"] = props.get("_auth_attempts", 0) + 1
        logger.debug("Waiting on authentication. Attempt %s" % props["_auth_attempts"])
        if props["_auth_attempts"] > 20:  # should we have this?
            raise ClientException(
                "Failed waiting for login via browser. Please retry and login via the browser pop-up window"
            )
        return 401

    if response.status_code == 401:
        try:
            err_resp = response.json()
        except json.JSONDecodeError:
            err_resp = None
        unauth_message = err_resp.get("error", {}).get("message") if err_resp else ""
        check_login_response(props, err_resp, logger)
        if no_vip_included(props):
            if unauth_message.lower() != "user must login":
               raise ClientException(
                   f"Failed to connect: {unauth_message}"
               )
        else:
            if "OpenIdProviderIp" not in props or props["OpenIdProviderIp"] == "":
                raise ClientException(
                    f"Failed to connect: {unauth_message}"
                )

        props["openid"] = binascii.b2a_hex(os.urandom(16)).decode("UTF-8")
        logger.info(f"User must authenticate via OIDC")
        props["token"] = ""
        props["refresh_token"] = ""
        if props.get("OpenIdCallbackHost") != "":
          remove_endpoint_from_host_file(env, props) #remove previous entry
          add_endpoint_to_host_file(env, props)
          #if not no_vip_included(props) and tunnel_all_traffic(props): #need to get to oidc provider via gateway
          #   add_oidc_provider_route(env, props)
          delete_tunnel_routes(env, props)
          logger.info(f"Starting OIDC authentication to {props['OpenIdCallbackHost']}")
          props['browser_supported'] = webbrowser.open(
            f"https://{props['OpenIdCallbackHost']}:8000/api/wg-connection/{clientpack_id}?state={props['openid']}",
            new=2,
          )
          if browser_not_supported(props):
              logger.info("Browser open failed, authentication url sent to console")
              message = f"  VNS3 requires completing authentication in your web browser.  Copy and past the following URL into your browser:\n\n   https://{props['OpenIdCallbackHost']}:8000/api/wg-connection/{clientpack_id}?state={props['openid']}"
              send_notification(env, message)
          return 401
        else:
          raise ClientException(
                "No OpenId hostname could be determined for the callback"
            )

    elif response.status_code == 200:
        if response.text == "null":
            logger.debug(
                "received empty response validating wireguard connection with controller, ignore"
            )
            return None
        resp = json.loads(response.text)
        if "vns3_version" in resp["response"]:
            props["vns3_version"] =  resp["response"]["vns3_version"]
            logger.debug(f"VNS3 version: {props['vns3_version']}")
        else:
            #when fail over to a controller that does not send version info
            props.pop("vns3_version", None)
        psk = resp["response"]["psk"]
        controller_vip = resp["response"]["controller_vip"]
        refresh_token = resp["response"]["refresh_token"]
        if controller_vip != "":
            props["ControllerVIP"] = controller_vip
        if refresh_token != "":
            props["refresh_token"] = refresh_token
        if psk != "":
            props["PresharedKey"] = psk
        else:
            props.pop("PresharedKey", None)
        props["wg_auth"] = json.loads(response.text)["response"]["auth"]
        #delete_oidc_provider_route(env, props)
        add_tunnel_routes(env, props, logger)
        if browser_not_supported(props):
            send_notification(env, "   Authentication successful, your Cohesive Network vpn connection is up")
        return 200

    elif response.status_code == 503:  #controller initializing, let's wait
        logger.debug("controller initializing")
        return None

    elif response.status_code == 400:
        try:
            err_resp = response.json()
        except json.JSONDecodeError:
            raise ClientException(error_msg)

        if err_resp.get("message") == "Invalid clientpack": #in case controller hasn't initialized yet
            return None

    else:
        error_msg = "connect failure: %s" % response.text
        logger.info(error_msg)
        remove_endpoint_from_host_file(env, props)
        raise ClientException(error_msg)


def wg_disconnect(endpoint, clientpack_id, api_token, logger=None, timeout=None):
    logger = logger or lib_logger
    logger.debug(f"Disconnecting from {endpoint} for clientpack {clientpack_id}")
    my_headers = {"api-token": api_token}
    try:
        response = requests.put(
            f"https://{endpoint}:8000/api/wg-connection/{clientpack_id}",
            headers=my_headers,
            verify=False,
            timeout=2, #don't wait too long if connection is down, 
                       #and especially if port 8000 is blocked
        )
    except Exception as e:
        return
    if response.status_code == 200:
        logger.debug("Disconnecting acknowledged %s" % clientpack_id)
    else:
        # Question better error message
        logger.info(
            f"Unexpected status code disconnecting [{response.status_code}]: {response.text}"
        )
    

def run_command_windows(command, logger=None, should_raise=False):
    process_kwargs = {
        "stdout": subprocess.PIPE,
        "stderr": subprocess.PIPE,
        "shell": False,
        "creationflags": subprocess.CREATE_NO_WINDOW
    }

    with subprocess.Popen(command, **process_kwargs) as proc:
        output, error = proc.communicate()
        if output is not None:
            output = output.decode("utf-8").strip()
        if error is not None:
            error = error.decode("utf-8").strip()

        if error:
            error_str = "Command stderr. command=%s: %s" % (command, error)
            if logger:
                logger.debug(error_str)
            proc.terminate()
            if should_raise:
                raise ClientException(error_str)
            return (output, error)
        return (output, error)


# utility method to run a system command
def run_command(command, logger=None, should_raise=False):
    process_kwargs = {
        "stdout": subprocess.PIPE,
        "stderr": subprocess.PIPE,
        "shell": True,
    }
    with subprocess.Popen(command, **process_kwargs) as proc:
        output, error = proc.communicate()
        if output is not None:
            output = output.decode("utf-8").strip()
        if error is not None:
            error = error.decode("utf-8").strip()

        if error:
            error_str = "Command stderr. command=%s: %s" % (command, error)
            if logger:
                logger.debug(error_str)
            proc.terminate()
            if should_raise:
                raise ClientException(error_str)
            return (output, error)
        return (output, error)


# return the last handshake time using wireguard system call
# def get_last_handshake(interface):
#     return run_command(
#         "wg show " + interface + " latest-handshakes | head -n1 | awk '{print $2;}'"
#     )


def get_last_handshake(env, interface):
    show_latest_handshakes = "%s show %s latest-handshakes" % (
        env.get("wg_path"),
        interface,
    )
    if env.is_windows:
        cmd1 = ["powershell", show_latest_handshakes]
        latest_handshakes, _ = run_command(cmd1)
        parse_cmd = "echo '%s'| %%{ $_.Split('=')[1]; }" % latest_handshakes
        command = ["powershell", parse_cmd]
        out, err = run_command_windows(command)    
    else:
        command = "%s | head -n1 | awk '{print $2;}'" % show_latest_handshakes
        out, err = run_command(command)

    if err and "permission" in err.lower():
        raise ClientException(
            "Invalid permissions for client. Cannot query OS interfaces. Requires root/admin permissions."
        )
    return 0 if out=='' else int(out)


# return the date from a specified number of seconds ago
def get_check_date(env, timeout):
    if env.is_linux:
        out, err = run_command(f"date -d '-{timeout} seconds' '+%s'")
    elif env.is_macos:
        out, err = run_command(f"date -v -{timeout}S +%s")
    elif env.is_windows:
        out, err = run_command_windows(
            ["powershell", f"(Get-Date).AddSeconds(-{timeout}) | Get-Date -UFormat %s"]
        )
    else:
        raise ClientException("Unknown platform %s" % env.platform)
    return out

def parse_time_duration(env, duration_str):
    if duration_str.strip().lower() == "now":
        duration_str = "1 second ago"
    # Use regular expression to extract days, hours, minutes, and seconds
    match = re.match(r'(?:(?:(\d+) days?,\s*)?(?:(\d+) hours?,\s*)?)?(?:(\d+) minutes?,\s*)?(\d+) seconds? ago', duration_str)
    if match:
        # Extract days, hours, minutes, and seconds
        days = int(match.group(1)) if match.group(1) else 0
        hours = int(match.group(2)) if match.group(2) else 0
        minutes = int(match.group(3)) if match.group(3) else 0
        seconds = int(match.group(4))
        return timedelta(days=days, hours=hours, minutes=minutes, seconds=seconds)
    else:
        # Check variations of "X minutes ago"
        minute_match = re.match(r'(\d+) minute(?:s)? ago', duration_str)
        if minute_match:
            minutes = int(minute_match.group(1))
            return timedelta(minutes=minutes)

        # Check variations of "X hour ago"
        hour_match = re.match(r'(\d+) hour(?:s)?(?:,\s*(\d+) minute(?:s)?)? ago', duration_str)
        if hour_match:
            hours = int(hour_match.group(1))
            return timedelta(hours=hours)

        # Check variations of "X day ago"
        day_match = re.match(r'(\d+) day(?:s)?(?:,\s*(\d+) hour(?:s)?(?:,\s*(\d+) minute(?:s)?)?)? ago', duration_str)
        if day_match:
            days = int(day_match.group(1))
            return timedelta(days=days)

    return None


def is_handshake_within_range(env, interface, handshake_timeout_interval):
    wg_show = "%s show %s" % (
        env.get("wg_path"),
        interface,
    )
    if env.is_windows:
        command = ["powershell", wg_show]
        wg_show_output, err = run_command_windows(command)    
    else:
        command = wg_show
        wg_show_output, err = run_command(command)

    if err and "permission" in err.lower():
        raise ClientException(
            "Invalid permissions for client. Cannot query OS interfaces. Requires root/admin permissions."
        )
    # use output, ex. latest handshake: 1 hour, 8 minutes, 38 seconds ago
    # to check if last handshake within handshake_timeout_interval
    match = re.search(r"latest handshake:\s*(.*)", wg_show_output)
    if match:
        handshake_time_str = match.group(1)
        handshake_time_delta = parse_time_duration(env, handshake_time_str)
        if handshake_time_delta:
            if handshake_time_delta <= timedelta(seconds=handshake_timeout_interval):
                env.logger.debug(f"Connection handshake in range ({handshake_timeout_interval}): {interface} handshake: {handshake_time_str}")
                return True
            else:
                env.logger.debug(f"Connection handshake out of range ({handshake_timeout_interval}): {interface} handshake: {handshake_time_str} ")
                return False
    else:
        env.logger.info(f"Client must disconnected. No latest handshake detected.")
        return False
    
    env.logger.info(
        f"No connection handshake for {interface} handshake: {handshake_time_str}"
    )
    return False

cached_handshake = 0

# using wg last handshake, determine if the interface needs to be reset
def needs_interface_reset(env, props, handshake_timeout_interval):
    global cached_handshake
    if not props["ActiveInterface"]:
        if env.state_file:
            file_utils.put_key_value(env.state_file, "status", "connecting")
        return True

    #check if wg inteface is up, if it is, only validate controller connection
    #if handshake exceeds uesr configured handshake_timout in wireguard config
    if wireguard.is_interface_up(env, props['ActiveInterface']):
        if props["type"] == ConnectionTypes.WIREGUARD_COHESIVE:
            if is_handshake_within_range(env, props["ActiveInterface"], handshake_timeout_interval):
                can_connect = True
            else:
                can_connect = verify_vns3_connectivity(env, props, None, env.logger)                
        
        else: #Native connection, no need to check controller
           can_connect = True

        if not can_connect:
            env.logger.info("No connectivity to controller, reset")
            props["ConnectionStatus"] = ""
            return True

        latest_handshake = get_last_handshake(env, props["ActiveInterface"])
        cached_handshake_plus_interval = cached_handshake + handshake_timeout_interval
        if cached_handshake_plus_interval < latest_handshake:
            if props["type"] == ConnectionTypes.WIREGUARD_COHESIVE:
                can_connect = wg_connect(
                    env, props, props["ClientpackId"], props["Secret"], env.logger
                )
                if can_connect == 401:
                    # connection has handshake, but user most likely has not authenticated
                    # no need to reset so return False
                    env.logger.info("Waiting for user to authenticate or secret failed")
                    return False
                
                if not can_connect:
                    env.logger.info("Could not validate controller connection")
                    return True
                
                
        
        cached_handshake = latest_handshake
        current_status = props["ConnectionStatus"]
        props["ConnectionStatus"] = "up"  
        # do logic if connection was down and now back up          
        if current_status != "up" :
            env.logger.info(f"Connection up on {props['ActiveInterface']}")
            if props['do_route_polling']:
                # mainly to reset routing agent
                start_route_polling(env, props)
            if env.state_file:
                file_utils.update_data_file(
                    env.state_file,
                    {
                        "status": "connected",
                        "pid": os.getpid(),
                        "last_connect": datetime.utcnow().strftime(
                            LibConfig.DATETIME_FORMAT
                        ),
                    },
                )
            if props["type"] == ConnectionTypes.WIREGUARD_NATIVE:
                env.logger.info(f"Connection up on {props['ActiveInterface']}") 
        remove_endpoint_from_host_file(env, props)
        return False

    props["ConnectionStatus"] = ""
    if env.state_file:
        file_utils.put_key_value(env.state_file, "status", "connecting")
    return True


# update the preshared key in the wg config file
def update_preshared_key_config(wg_conf_file, psk):
    try:
        wg_fin = open(wg_conf_file, "r")
        prev_psk = ""
        config = ""
        for line in wg_fin:
            if line.startswith("PresharedKey"):
                if psk == "":
                    line = ""
                else:
                    prev_psk = line.partition(" =")[2].strip()
                    line = line.replace(prev_psk, psk)
            config = config + line
    finally:
        wg_fin.close()

    if psk != "" and config.find("PresharedKey") == -1:
        config = config.replace(f"[Peer]", f"[Peer]\nPresharedKey = {psk}")
    with open(wg_conf_file, "w") as wg_fout:
        wg_fout.write(config)



# wg-quick with run_command returns error when it may not be
# for now, if there is no "error" in the message, assume no error
# would be good to think of a better way to handle this
def handle_wg_quick_output(stdout, err, logger=None):
    if err and "error" in err.lower():
        logger.info(f"wg-quick error: {err}")
        raise ClientException(err)
    return (stdout, None)


def display_interface_status(env, interface, timeout):
    logger = env.logger
    if not interface:
        return
    if wireguard.is_interface_down(env, interface):
        logger.debug(f"{interface} interface is down")
        return

    lhs = get_last_handshake(env, interface)
    #if lhs == 0:
    #    logger.info(f"{interface} must be down, no handshake {lhs}")
    #else:
    #    if env.is_windows:
    #        check_hs, err = run_command_windows(
    #            [
    #                "powershell",
    #                f"{env.get('wg_path')} show {interface} | Select-String -Pattern 'latest handshake'"
    #            ]
    #        )
    #    else:
    #        check_hs, err = run_command(
    #            f"{env.get('wg_path')} show {interface} | grep 'latest handshake'"
    #        )
        #logger.debug(
        #    f"Wireguard {interface} is up, {check_hs} (reset if no handshake within: {timeout} seconds)"
        #)

def parse_address(props):
    parts = props['Address'].split(",")
    # Extract the value before '/' if '/' exists
    results = [re.split(r"/", part.strip())[0] for part in parts if "/" in part]
    return results[0]

def parse_clientpack(clientpack_path_or_lines):
    """parse_clientpack

    Parse file for clientpack configuration details. Currently only wiregaurd
    is supported

    Args:
        clientpack_path_or_lines (str|List[str]): path to clientpack file or list of clientpack lines

    Returns:
        dict: data
    """
    data = {}
    if type(clientpack_path_or_lines) is str:
        with open(clientpack_path_or_lines, "r") as cp_file:
            clientpack_lines = cp_file.readlines()
        data["wg_cp_file"] = clientpack_path_or_lines
        data["interface"] = os.path.basename(clientpack_path_or_lines).split(".")[0]
    else:
        clientpack_lines = clientpack_path_or_lines

    for line in clientpack_lines:
        if "=" in line:
            toks = line.partition("=")
            token = toks[0].strip()
            if token == "#Endpoint" or token == "#MTU" or token == "#DNS":
                continue  # ignore these tokens
            if token.startswith("#"):
                token = token[1:]
            data[token] = toks[2].strip()

    data["openid"] = ""
    data["token"] = ""
    data["refresh_token"] = ""
    data["ConnectionStatus"] = ""

    if "Address" in data:
        data["Address"] = parse_address(data)
        data["ClientpackId"] = data["Address"].replace(".", "_")

    return data


def wg_allowed_ips(env, routes, props):
    env.logger.debug(f"add allowed ips for {routes}")
    if routes is None:
        routes = []
    append_routes = ""
    for r in set(routes):
        append_routes += ", " + f"{r[0]}/{r[1]}"
    env.logger.debug(f"adding allowed ips: {env.get('wg_path')} set {props['ActiveInterface']} peer {props['PublicKey']} allowed-ips \'{props['AllowedIPs']}{append_routes}\'")
    set_cmd = f"{env.get('wg_path')} set {props['ActiveInterface']} peer {props['PublicKey']} allowed-ips \"{props['AllowedIPs']}{append_routes}\""
    if env.is_windows:
        command = ["powershell", set_cmd]
        out, err = run_command_windows(command)
    else:
        command = set_cmd
        out, err = run_command(command)

    return out


def ping_controller_via_overlay(env, controller, clientpack, timeout=None):
    try:
        response = requests.get(
            f"https://{controller}:8000/api/status/connected_subnets?cp={clientpack}",
            headers={},
            auth=HTTPBasicAuth("overlayapi", "x"),
            verify=False,
            timeout=(timeout or LibConfig.DEFAULT_CONNECT_TIMEOUT),
        )
        return response.status_code == 200
    except Exception as e:
        env.logger.info(
            "Failed to ping controller %s via overlay. Error=%s" % (controller, str(e))
        )
        return False

def validate_route(host, mask):
    ipaddress.ip_network(f"{host}/{mask}", strict=False)

def update_routes(env, props, routes_file, server, interface, cp, src_cidr):
    if props["ConnectionStatus"] != "up":
        return
    src_address = src_cidr.split("/")[0]
    new_routes = vpn3api_desc_connected_subnets(env, props, server, cp)
    if new_routes is None:
        # no change to routes
        return
    routes = []
    for route in open(routes_file).readlines():
        a, b = route.replace("(", "").replace(")", "").split(",")
        routes.append((a.strip().replace("'", ""), int(b.strip().replace("'", ""))))

    if new_routes != routes:
        routes_to_add = set(new_routes).difference(routes)
        routes_to_del = set(routes).difference(new_routes)
        env.logger.info(
            "Route change detected. Add=%s Del=%s" % (routes_to_add, routes_to_del)
        )
        wg_allowed_ips(env, new_routes, props)
        for r in routes_to_add:
            if is_valid_ip_route(r, env): 
                add_route(env, r, interface, src_address)
        for r in routes_to_del:
            if is_valid_ip_route(r, env):
                del_route(env, r, interface)
        routes = new_routes

    with open(routes_file, "w") as outfile:
        outfile.write("\n".join(str(item) for item in new_routes))


def delete_dynamic_routes(env, props):
    routes_file = props.get("routes_file")
    if routes_file:
        try:
            fileroutes = open(routes_file)
            for route in fileroutes.readlines():
                a, b = route.replace("(", "").replace(")", "").split(",")
                route = (
                    a.strip().replace("'", ""),
                    b.strip().replace("'", ""),
                )
                # Question can be interface from props?
                del_route(env, route, props["ActiveInterface"])
        finally:
            fileroutes.close()
        
        


def delete_all_routes(env, props):
    delete_dynamic_routes(env, props)
    delete_local_routes(env, props)
    delete_endpoint_routes(env, props)
    delete_tunnel_routes(env, props)
    #delete_oidc_provider_route(env, props)


def vpn3api_desc_connected_subnets(env, props, controller, cp, timeout=None):
    """
    Returns None when an error was encountered; returns
    a sorted list of route tuples otherwise
    """
    try:
        endpoint = get_current_endpoint(props)
        env.logger.debug("Syncing routes from %s" % endpoint)
        response = requests.get(
            f"https://{controller}:8000/api/status/connected_subnets?cp={cp}",
            headers={},
            auth=HTTPBasicAuth("overlayapi", "x"),
            verify=False,
            timeout=(timeout or LibConfig.DEFAULT_CONNECT_TIMEOUT),
        )
        if response.status_code != 200:
            env.logger.debug(
                "Fetching routes failed (%s): %s"
                % (response.status_code, response.text)
            )
            return None

        routes = response.json().get("response")
        for r in routes:
            r[1] = netmask_to_cidr(r[1])
            validate_route(r[0], r[1])
        env.logger.debug("Fetched and validated routes: %s" % routes)
        return sorted([tuple(r) for r in routes])
    except Exception as e:
        env.logger.debug(
            "Fetching routes failed. Controller=%s Exception=%s" % (controller, str(e))
        )
        return None

def netmask_to_cidr(m_netmask):
    try:
        if "." in m_netmask:
            # IPv4
            return sum(bin(int(part)).count("1") for part in m_netmask.split("."))
        elif ":" in m_netmask:
            # IPv6
            addr = ipaddress.IPv6Address(m_netmask)
            bits = ''.join(f'{byte:08b}' for byte in addr.packed)
            return bits.count('1')
        else:
            raise ValueError("Invalid netmask format")
    except Exception as e:
        raise ValueError(f"Failed to convert netmask '{m_netmask}' to CIDR: {e}")


def netmask_to_cidr2(m_netmask):
    return sum([bin(int(bits)).count("1") for bits in m_netmask.split(".")])

def cidr_to_netmask(cidr):
    network, net_bits = cidr.split("/")
    host_bits = 32 - int(net_bits)
    netmask = socket.inet_ntoa(struct.pack("!I", (1 << 32) - (1 << host_bits)))
    return netmask

def get_ip_flag(env, route):
    if ":" in route:
        if env.is_linux:
            return "-6"
        elif env.is_macos:
            return "-inet6"
    else:
        if env.is_macos:
            return "-net"
    return ""

def get_default_interface(env, props, route):
    dev = None
    flag = get_ip_flag(env, route)
    if env.is_linux:
        dev, error = run_command(f"ip {flag} route show default | awk '/default/ {{print $5}}'")	
    elif env.is_macos:
        dev, error = run_command(f"route -n get -inet6 default | awk '/interface:/ {{print $2}}'")
    return dev

def ip_add_route_command(env, props, route):
    env.logger.info(f"add route: {route}")
    dev = get_default_interface(env, props, route)
    gateway, err = get_default_gateway(env, props, route)
    env.logger.info(f"default interface:{dev} gateway:{gateway}")
    if env.is_linux:
        if is_ipv6(route):
            return f"ip -6 route add {route} via {gateway} dev {dev}"
        else:
            return f"ip route add {route} via {gateway} dev {dev}"
    elif env.is_macos:
        if is_ipv6(route):
            return f"route -n add -inet6 {route} {gateway}%{dev}"
        else:
            return f"route -n add -net {route} {gateway}" 
    elif env.is_windows:
        if is_ipv6(route):
            return f"netsh interface ipv6 add route {route} {props['ActiveInterface']} {gateway}"
        else:
            return f"route ADD {route} MASK 255.255.255.255 {gateway} metric 1"
    return None

def add_local_routes(env, props, logger=None):
    if "LocalRoutes" in props and props["LocalRoutes"] != "":
        routes = props["LocalRoutes"].split(",")
        for route in routes:
            (logger or lib_logger).debug(f"Adding local route: {route}")
            if env.is_linux:
                cmd = ip_add_route_command(env, props,route)
                run_command(ip_add_route_command(env, props,route))

            elif env.is_macos:
                cmd = ip_add_route_command(env, props,route)
                run_command(ip_add_route_command(env, props,route))

            else:
                run_command_windows(ip_add_route_command(env, props,route))

def add_tunnel_routes(env, props, logger=None):
    if tunnel_all_traffic(props) and props["ActiveInterface"] is not None:
        logger.info(f"Tunnel all traffic to {props['ActiveInterface']}")
        
        allowed_ips = f"0.0.0.0/1, 128.0.0.0/1, {props['AllowedIPs']}"
        ipv6 = is_ipv6(props["Endpoint"]) 
        if ipv6 and "::/0" not in allowed_ips:
            allowed_ips ="::/0," + allowed_ips #ipv6

        command = f"{env.get('wg_path')} set {props['ActiveInterface']} peer {props['PublicKey']} allowed-ips '{allowed_ips}'"
        if env.is_windows:
            add_route(env, ["128.0.0.0", "1"],  props["ActiveInterface"], props["Address"])
            add_route(env, ["0.0.0.0", "1"], props["ActiveInterface"], props["Address"])
            if ipv6:
                add_route(env, ["::", "0"], props["ActiveInterface"], props["Address"])
            win_command = ["powershell", command]
            run_command_windows(win_command, env.logger)
        else:
            add_route(env, ["128.0.0.0", "1"],  props["ActiveInterface"], props["Address"])
            add_route(env, ["0.0.0.0", "1"], props["ActiveInterface"], props["Address"])
            if ipv6:
                add_route(env, ["::", "0"], props["ActiveInterface"], props["Address"])
            run_command(command)

def delete_tunnel_routes(env, props):
    if tunnel_all_traffic(props):
        allowed_ips = props['AllowedIPs']
        command = f"{env.get('wg_path')} set {props['ActiveInterface']} peer {props['PublicKey']} allowed-ips '{allowed_ips}'"
        if env.is_windows: 
            try:
                address = props['Address'].split('/')[0]
                route1 = f"Get-NetRoute -DestinationPrefix 0.0.0.0/0 -NextHop {address} | Remove-NetRoute"
                run_command_windows(route1, env.logger)
                route2 = f"Get-NetRoute -DestinationPrefix 128.0.0.0/0 -NextHop {address} | Remove-NetRoute"
                run_command_windows(route2, env.logger)
            except Exception as e:
                env.logger.debug("delete tunnel routes ex: %s", str(e))
            win_command = ["powershell", command]
            run_command_windows(win_command, env.logger)
        else:
            del_route(env, ["128.0.0.0", "1"], props["ActiveInterface"])
            del_route(env, ["0.0.0.0", "1"], props["ActiveInterface"])
            run_command(command, env.logger)
        
def delete_local_routes(env, props):
    if "LocalRoutes" in props and props["LocalRoutes"] != "":
        routes = props["LocalRoutes"].split(",")
        for route in routes:
            env.logger.debug(f"Removing local route: {route}")
            if env.is_linux:
                run_command(f"ip route delete {route}")
            elif env.is_macos:
                 run_command(f"route delete {route}")
            else:
                cmd = f"route delete {route}"
                run_command_windows(cmd, env.logger)


def get_wg_adapter_index(interface):
    out, err = run_command(
        [
            "powershell",
            "Get-NetAdapter -Name {interface} | select -ExpandProperty ifIndex -first 1".format(interface=interface),
        ]
    )
    return out

def add_oidc_provider_route(env, props):
    if not "OpenIdProviderIp" in props or props["OpenIdProviderIp"] == "":
        return
    def_log_message = (
        f"Add local route for oidc provider"
    )
    #env.logger.debug(f"{def_log_message}")
    # this is done in get_default_gateway and so i dont believe its necessary here
    # controller = get_current_endpoint(props)
    # if controller is None:
    #     env.logger.info(
    #         f"{def_log_message} failed: could not determine controller endpoint"
    #     )
    #     return

    # find the local gateway address
    gateway, err = get_default_gateway(env, props)
    if err or gateway is None:
        message = err or 'Could not determine gateway address'
        env.logger.info(
            f"{def_log_message} failed: cound not get default gateway: {message}"
        )
        return

    endpoint = props["OpenIdProviderIp"]
    # find platform specific gateway command
    if env.is_linux:
        command = f"ip route add {endpoint}/32 via {gateway}"

    elif env.is_macos:
        command = f"route add -net {endpoint}/32 {gateway}"

    elif env.is_windows:
        command = f"route ADD {endpoint} MASK 255.255.255.255 {gateway} metric 1"

    else:
        env.logger.info(f"{def_log_message} failed: no platform support")
        return

    # add the controller route
    env.logger.info(f"{def_log_message} {endpoint}")
    response, err = run_command(command)
    if err:
        env.logger.info(f"{def_log_message} failed: {err}")

    if "OpenIdProviderIp" in props and "OpenIdProviderHost" in props:
        add_hosts_entry(env, props['OpenIdProviderIp'], props['OpenIdProviderHost'])

def is_ipv6(address):
    return bool(address) and address.count(":") > 1

def add_controller_routes(env, props, gateway=None):
    def_log_message = (
        f"Add local routes for controller"
    )
    # this is done in get_default_gateway and so i dont believe its necessary here
    # controller = get_current_endpoint(props)
    # if controller is None:
    #     env.logger.info(
    #         f"{def_log_message} failed: could not determine controller endpoint"
    #     )
    #     return

    # find the local gateway address
    if gateway is None:
        gateway, err = get_default_gateway(env, props)
        if err or gateway is None:
            message = err or 'could not determine gateway address'
            env.logger.debug(
                f"{def_log_message} failed: cound not get default gateway: {message}"
            )
            return err
    for endpoint in get_secondary_endpoints_props(props):
        if is_ipv6(endpoint):

            endpoint= endpoint+"/128"
        else:
            endpoint= endpoint+"/32"
        if env.is_linux:
            run_command(f"ip route delete {endpoint}")
        else:
            run_command(f"route delete {endpoint}")
        # find platform specific gateway command
        if env.is_linux:
            command = ip_add_route_command(env, props, endpoint)

        elif env.is_macos:
            command = ip_add_route_command(env, props, endpoint)

        elif env.is_windows:
            command = ip_add_route_command(env, props, endpoint)

        else:
            env.logger.info(f"{def_log_message} failed: no platform support")
            return

        # add the controller route
        env.logger.debug(f"{def_log_message} {endpoint} via {gateway}")
        response, err = run_command(command)
        if err and not "exists" in err:
            env.logger.info(f"{def_log_message} failed: {err}")
            return err

# address is used to determine ipv4 or ipv6 gateway
# if not provided, use the current endpoint
def get_default_gateway(env, props, address=None):
    def_log_message = "get default gateway"
    if address is None:
        address = get_current_endpoint(props)
        if address is None:
            return (
            None,
            "{msg} failed: Controller endpoint could not be determined".format(msg=def_log_message),
        )
        if is_ipv6(address):
            address = address.replace("[","").replace("]","")

    ip_flag = get_ip_flag(env, address) 
    if env.is_linux:
        gateway, err = run_command(
            f"ip {ip_flag} route show default | awk '/default/ {{print $3}}'"
        )

    elif env.is_macos:
        gateway, err = run_command(
            f"route -n get {ip_flag} default | awk '/gateway/ {{print $2}}' | cut -d'%' -f1"
        )

    elif env.is_windows:
        if is_ipv6(address):
            gateway, err = run_command_windows(
            [
                "powershell",
               '(Get-NetRoute -AddressFamily IPv6 | Where-Object { $_.NextHop -ne "::" } | Select-Object -First 1).NextHop'
            ]
        )
        else:
            gateway, err = run_command_windows(
                [
                    "powershell",
                '(Get-NetRoute -AddressFamily IPv4 | Where-Object { $_.NextHop -ne "0.0.0.0" } | Select-Object -First 1).NextHop'
                ]
            )

    else:
        return (None, "{msg}: no platform support found".format(msg=def_log_message))

    if err:
        env.logger.debug(f"error fetching local gateway: {err}")
    return (gateway, err)


def delete_oidc_provider_route(env, props):
   if "OpenIdProviderIp" in props:
       if env.is_linux:
           run_command(f"ip route delete {props['OpenIdProviderIp']}")
       else:
           run_command(f"route delete {props['OpenIdProviderIp']}")
       env.logger.debug(f"removed oidc route: {props['OpenIdProviderIp']}")
   #remove entry in hosts file
   if "OpenIdProviderIp" in props and "OpenIdProviderHost" in props:
       remove_hosts_entry(env, props['OpenIdProviderIp'], props['OpenIdProviderHost'])


def delete_endpoint_routes(env, props):
    for route in get_secondary_endpoints_props(props):
        if ":" in route:
            route = route.replace("[","").replace("]","")
        else:
            route = os_utils.resolve_ip(route)

        if env.is_linux:
            run_command(f"ip route delete {route}")
        if env.is_posix:
            run_command(f"route delete {route}")
        else:
            run_command_windows(f"route delete {route}", env.logger)

def is_valid_ipv4_route(route, env):
    try:
        ipaddress.IPv4Network(route, strict=False)
        return True
    except ValueError:
        env.logger.info(f"Route format check failed: {route} {time.asctime()}")
        return False

def is_valid_ipv6_route(route, env):
    try:
        ipaddress.IPv4Network(route, strict=False)
        return True
    except ValueError:
        env.logger.info(f"Route format check failed: {route} {time.asctime()}")
        return False

def is_valid_ip_route(route, env):
    try:
        # Try parsing as either IPv4 or IPv6 network
        ipaddress.ip_network(route, strict=False)
        return True
    except ValueError:
        env.logger.info(f"Route format check failed: {route} {time.asctime()}")
        return False

def add_route(env, route, interface, src_address):
    env.logger.debug(f"Adding route: {route}")
    if env.is_posix:
        try:
            ip_flag = get_ip_flag(env, route[0]) 
            metric_flag = "metric 50" if route[0]=="::" else ""
            if env.is_linux:
                cmd = f"ip {ip_flag} route add {route[0]}/{route[1]} dev {interface} {metric_flag}"
            else:
                cmd = f"route add {ip_flag} {route[0]}/{route[1]} -interface {interface}"
            run_command(cmd, env.logger)
        except Exception as e:
            env.logger.info({e})
    else:
        # gw = get_overlay_gateway_on_windows(interface)
        # if gw is None:
        #     env.logger.info("bad gateway %s", time.asctime())
        #     return
        # cmd = "route add %s mask %s " % route
        # cmd += "%s >NUL 2>&1" % gw
        # os.system(cmd)

        # TODO:changed: src_address should be this parsed from Address
        # gw = props['Address'].rpartition("/")[0]
        try:
            gw = src_address.split('/')[0]
            if is_ipv6(route):
                run_command_windows(f"netsh interface ipv6 add route {route} {interface} {gw}", env.logger)
            else:
                index = get_wg_adapter_index(interface)
                mask = cidr_to_netmask(route[0] + "/" + str(route[1]))
                cmd = f"route ADD {route[0]} MASK {mask} {gw} METRIC 1 IF {index}"
                run_command_windows(cmd, env.logger)
        except Exception as e:
            env.logger.info({e})



def del_route(env, route, interface):
    env.logger.debug(f"Removing route: {route} {time.asctime()}")
    if env.is_posix:
        if env.is_linux:
            cmd = f"ip route del {route[0]}/{route[1]} dev {interface}"
        else:
            cmd = f"route delete -net {route[0]}/{route[1]} -interface {interface}"
        run_command(cmd, env.logger) 
    else:
        cmd = f"route delete {route[0]}"
        run_command_windows(cmd, env.logger)


def get_overlay_gateway_on_linux(interface):
    f = os.popen(f"ifconfig {interface}")
    gw = None
    for line in f:
        if "P-t-P" in line:
            parts = line.split()
            for p in parts:
                if "P-t-P" in p:
                    gw = p[6:]
    f.close()
    return gw


def initialize(env, props):
    wg_cp_file = props["wg_cp_file"]
    if path.exists(wg_cp_file) == False:
        error = f"Could not find clientpack file: {wg_cp_file}"
        env.logger.error(error)
        raise ClientException(error)
    props["wg_auth"] = env.get("wg_auth")
    clear_routes_file(env, props)


def clear_routes_file(env, props):
    routes_file = props.get("routes_file")
    if not routes_file:
        return
    props["routes_file"] = routes_file
    # Keep previous routes fetched to compare if new routes occurred
    if env.is_windows:
        if not os.path.exists(routes_file):
            Path(routes_file).write_text("")
    else:
        Path(routes_file).touch()
    Path(routes_file).write_text("")

def get_endpoint_for_config_file(props):
    endpoint = props["Endpoint"]

    # Split the endpoint into address and port
    if endpoint.startswith("["):  # already wrapped (IPv6 with port)
        address, port = endpoint[1:].split("]")[0], endpoint.split("]:")[-1] if "]:" in endpoint else ""
    else:
        parts = endpoint.rsplit(":", 1)
        if len(parts) == 2 and parts[1].isdigit():
            address, port = parts
        else:
            address, port = endpoint, ""

    # Determine if IPv6 and wrap if needed
    is_ipv6_endpoint = ":" in address and not address.startswith("[")
    formatted_address = f"[{address}]" if is_ipv6_endpoint else address

    # Recombine address and port
    update = f"{formatted_address}:{port}" if port else formatted_address
    return update;

def update_wg_config(env, props, logger=None):
    logger = logger or lib_logger
    wg_conf_file = props["wg_conf_file"]

    logger.debug(
        f"Updating wireguard config file: {wg_conf_file} with cohesive functionality if exists"
    )

    # custom cohesive stuff
    tunnel_is_gateway = props.get(
        "TunnelAllTraffic", LibConfig.Defaults.TUNNEL_ALL_TRAFFIC
    )
    has_secondary_endpoints = len(get_secondary_endpoints_props(props)) > 0
    check_tunnel_all_traffic = (
        tunnel_is_gateway
        or "128.0.0.0/1" in props["AllowedIPs"]
        or "0.0.0.0/1" in props["AllowedIPs"]
    )

    has_cohesive_edits = check_tunnel_all_traffic or has_secondary_endpoints
    if has_cohesive_edits:
        try:
            wg_fin = open(wg_conf_file, "r")
            final_conf_lines = []

            # if line needs to be overwritten, append and continue.
            for line in wg_fin:
                if line.startswith("AllowedIPs"):
                    if "0.0.0.0" in props["AllowedIPs"] or "128.0.0.0/1" in props["AllowedIPs"]:
                        props["AllowedIPs"] = (
                            props["AllowedIPs"]
                            .replace(",", " ")
                            .replace("0.0.0.0/1", " ")
                            .replace("128.0.0.0/1", " ")
                        )

                        props["AllowedIPs"] = ", ".join(props["AllowedIPs"].split())
                        final_conf_lines.append(f"AllowedIPs = {props['AllowedIPs']}")
                        continue

                if line.startswith("Endpoint"):
                    wg_endpoint = get_endpoint_for_config_file(props)
                    final_conf_lines.append(f"Endpoint = {wg_endpoint}")
                    continue

                if line.startswith("#SecondaryEndpoints") and has_secondary_endpoints:
                    secondaryEndpoints = props.get("SecondaryEndpoints","")
                    final_conf_lines.append(f"#SecondaryEndpoints = {secondaryEndpoints}")
                    continue

                elif line.startswith("SaveConfig") and env.is_windows:
                    # SaveConfig not supported on windows
                    continue

                #elif line.startswith("#SecondaryEndpoints"):
                #    endpoint = get_current_endpoint(props)
                #    endpoints = [
                #        x.strip()
                #        for x in props["SecondaryEndpoints"].split(",")
                #        if x.strip() != endpoint
                #    ]
                #    secondaryEndpoints = ", ".join(endpoints)
                #    final_conf_lines.append(f"#SecondaryEndpoints = {secondaryEndpoints}")
                #    continue

                final_conf_lines.append(line.strip())
        finally:
            wg_fin.close()
            
        # overwrite with final conf
        with open(wg_conf_file, "w") as wg_fout:
            wg_fout.write("\n".join(final_conf_lines))
            os.chmod(
                wg_conf_file, wireguard.CONF_FILE_CHMOD
            )  # user read/write, wg spits out warning if public access


def find_and_update_interface_by_overlay_ip(env, props, logger=None):
    """find_and_update_interface_by_overlay_ip

    Searches interfaces by IP for interface with clientpacks overlay ip. If it
    finds one, it updates ActiveInterface prop. SHould really only be called on macos

    Args:
        env (_type_): _description_
        props (_type_): _description_
        logger (_type_, optional): _description_. Defaults to None.

    Returns: Boolean or str
    """
    if logger: logger.debug("Searching for active interface for IP %s" % props["overlay_ip"])
    actual_interface = os_utils.find_ip_interface(
        env, props["overlay_ip"], retries=5, sleep=0.5
    )
    if actual_interface:
        if logger: logger.debug(f"Changing interface to {actual_interface}")
        props["ActiveInterface"] = actual_interface
        if env.state_file:
            file_utils.put_key_value(env.state_file, "interface", actual_interface)
        return actual_interface
    return None


def no_vip_included(props):
    return props["NoVIPIncluded"] == True

def start_wg_interface(env, props, interface, endpoint_selected, logger):
    data, interface_up_error = wireguard.wg_quick_up(
        env, props["ActiveInterface"], props["wg_conf_file"]
    )
    if interface_up_error:
        logger.debug(f"Failed to bring up wg interface: {data} {interface_up_error}")
    else:
        # only macos
        if interface is None:
            interface = find_and_update_interface_by_overlay_ip(env, props, logger)

        iface_details, iface_err = wireguard.get_interface_details(
            env, env.get("wg_path"), props["ActiveInterface"],
        retry_attempts=(12 if env.is_windows else 5), retry_sleep=0.8)
        if iface_err: # should we check on interface details?
            logger.info(f"Failed wg connect to VNS3 controller: {endpoint_selected} {iface_err}")
            return False
        else:
            logger.debug(f"Interface initialized: {iface_details}")
            return True

def reset_interface(env, interface, props, logger=None):
    global cached_handshake
    logger = logger or lib_logger
    logger.debug(f"Reset wireguard connection {interface or props['ClientpackId']}")
    props["ConnectionStatus"] = ""
    if props["type"] == ConnectionTypes.WIREGUARD_COHESIVE:
        clear_routes_file(env, props)
        endpoint_candidates = get_endpoint_retry_order(props)
        logger.debug("Trying to connect via endpoints: %s" % endpoint_candidates)
        can_connect = False
        interface_up_error = None
        endpoint_selected = None
        while (len(endpoint_candidates) > 0 and not can_connect):           
            interface = props['ActiveInterface']
            interface_up_error = None
            if interface != None:
               wireguard.wg_quick_down(env, interface, props["wg_conf_file"])

            # get next endpoint candidate
            endpoint_selected = endpoint_candidates.pop(0)
            logger.info(f"Connecting to controller: {endpoint_selected}")
            props = update_props_endpoint(props, endpoint_selected)
            
            update_wg_config(env, props, logger)
            
            if no_vip_included(props):
                # if not vip we do not bring up wg connection first
                can_connect = True

            else:
                can_connect = start_wg_interface(env, props, interface, endpoint_selected, logger)
                cached_handshake = 0
                error = add_controller_routes(env, props)
                if error:
                    return False

            if can_connect:
                update_openid_callback_host(props, endpoint_selected)
                vns3_connect_success = wg_connect(
                    env, props, props["ClientpackId"], props["Secret"], logger=logger
                )
                if vns3_connect_success == 401:
                    # controller responded with waiting for auth
                    env.logger.info("Waiting on user to authenticate")
                    can_connect = True
                    continue

                if not vns3_connect_success:
                    # could not connect to controller
                    can_connect = False
                    if env.state_file:
                        file_utils.update_data_file(
                            env.state_file,
                            {
                                "status": "connecting",
                                "pid": os.getpid(),
                                "last_connect": datetime.utcnow().strftime(
                                    LibConfig.DATETIME_FORMAT
                                ),
                            },
                    )
                    continue
                
                if not no_vip_included(props):
                    # success from wg_connect means connection is up
                    cached_handshake = get_last_handshake(env, props["ActiveInterface"])
                    if env.state_file:
                        file_utils.update_data_file(
                            env.state_file,
                            {
                                "status": "connected",
                                "pid": os.getpid(),
                                "last_connect": datetime.utcnow().strftime(
                                    LibConfig.DATETIME_FORMAT
                                ),
                            },
                        )
                    env.logger.info(f"Connection is up on {props['ActiveInterface']}")

                delete_local_routes(env, props)
                add_local_routes(env, props, logger=env.logger)

                if no_vip_included(props):
                    can_connect = start_wg_interface(env, props, interface, endpoint_selected, logger)
                    if env.state_file:
                        file_utils.update_data_file(
                            env.state_file,
                            {
                                "status": "connected",
                                "pid": os.getpid(),
                                "last_connect": datetime.utcnow().strftime(
                                    LibConfig.DATETIME_FORMAT
                                ),
                            },
                        )
                    cached_handshake = 0
                    if can_connect:
                        cached_handshake = get_last_handshake(env, props["ActiveInterface"])
                        add_tunnel_routes(env, props, logger)
                        if cached_handshake:
                            env.logger.info(f"Connection now up on {props['ActiveInterface']}")
                        else:
                            can_connect = False
                            continue

                if props["ConnectionStatus"] == "":
                    if "PresharedKey" in props:
                        update_preshared_key_config(
                            props["wg_conf_file"], props["PresharedKey"]
                        )
                    else:
                        update_preshared_key_config(props["wg_conf_file"], "")
            
        if not can_connect:
            logger.info(f"Could not connect, resetting and trying again")
            return False

    else: # wireguard native
        wireguard.wg_quick_down(env, interface, props["wg_conf_file"])
        _, interface_up_error = wireguard.wg_quick_up(
            env, props["ActiveInterface"], props["wg_conf_file"]
        )

        if interface_up_error:
            # This could be a fatal error. Should we throw here?
            logger.error("Failed to up interface: %s" % interface_up_error)
            return False

        if interface is None: # have to call again for 
            interface = find_and_update_interface_by_overlay_ip(env, props, logger)

    interface_details, err = wireguard.get_interface_details(
        env, env.get("wg_path"), props["ActiveInterface"],
        retry_attempts=(10 if env.is_windows else 1), retry_sleep=1.0
    )
    # allow does not exist due to possible race condition.
    # but fail on path path and permissions errors
    if err and "does not exist" not in err:
        raise ClientException("Failed to get interface details after up: %s" % str(err))
    logger.debug("Interface details: %s" % interface_details)
    
    return True


def validate_clientpack(props, logger=None, validate_endpoints=False):
    """validate_clientpack - this DOES mutate props. Output is a valid clientpack
       dictionary using standard python data types

    Args:
        props (dict): dict parsed from clientpack file
        logger (optional): Defaults to library logger.
        validate_endpoints (boolean): Default: True. If True, it will query endpoints for first available

    Returns:
        dict: Always returns dict. empty if none. otherwise errors keyed by type:
        {
            'wg-native': List[str] # list of errors for wg native type,
            'wg-cohesive': List[str] # list of errors for wg cohesive type,
        }
    """
    logger = logger or lib_logger
    connection_type = props.get("type")
    wg_native_errors = []
    wg_cn_errors = []
    if "Address" not in props:
        wg_native_errors.append(
            f"Address must be specified and be a network cidr in conf file"
        )
    else:
        try:
            ipv4_pattern = r"\b\d{1,3}(?:\.\d{1,3}){3}(?:/\d{1,2})?\b"
            ipv6_pattern = r"\b(?:[0-9a-fA-F]{0,4}:){2,7}[0-9a-fA-F]{0,4}(?:/\d{1,3})?\b"

            ipv4_addresses = re.findall(ipv4_pattern, props["Address"])
            ipv6_addresses = re.findall(ipv6_pattern, props["Address"])

            # prioritize IPv4 addresses if both are present
            if ipv4_addresses:
                props["overlay_ip"] = ipv4_addresses[0].split("/")[0]
            elif ipv6_addresses:
                props["overlay_ip"] = ipv6_addresses[0].split("/")[0]
            
        except ValueError:
            logger.info(f"AllowedIPs must be specified in a network cidr format: {route}")

    if "PrivateKey" not in props:
        wg_native_errors.append(f"PrivateKey must be specified in conf file")

    if "PublicKey" not in props:
        wg_native_errors.append(f"PublicKey must be specified in conf file")

    if "AllowedIPs" not in props:
        wg_native_errors.append(
            f"AllowedIPs must be specified and be a network cidr in conf file"
        )
    else:
        for route in props["AllowedIPs"].split(","):
            if not re.match(
                wireguard.CIDR_CHECK_REGEX,
                route.strip(),
            ) and not ":" in route:
                wg_native_errors.append(
                    f"AllowedIPs must be specified in a network cidr format: {route}"
                )

    if "SaveConfig" in props:
        props["SaveConfig"] = string_to_boolean(props["SaveConfig"])
        if (
            props["SaveConfig"] is True
            and connection_type == ConnectionTypes.WIREGUARD_COHESIVE
        ):
            wg_cn_errors.append(
                "SaveConfig=true is not supported for cohesive wireguard connection type. Please use native connection type to set SaveConfig=true"
            )

    if "Secret" not in props:
        props["Secret"] = ""
        logger.info(f"Secret must be specified in conf file to use VNS3 Wireguard vpn connections")

    if "TunnelAllTraffic" in props:
        tunnel_all_traffic_val = string_to_boolean(props["TunnelAllTraffic"])
        if tunnel_all_traffic_val is None:
            wg_cn_errors.append(
                f"TunnelAllTraffic misconfigured, must be True or False"
            )
        else:
            props["TunnelAllTraffic"] = tunnel_all_traffic_val
            logger.info(
                f"TunnelAllTraffic configured to '{tunnel_all_traffic_val}'"
            )
    else:
        props["TunnelAllTraffic"] = False
        logger.info(
            f"TunnelAllTraffic not configured, can be True or False, setting to '{props['TunnelAllTraffic']}'"
        )

    if "RoutePolling" in props:
        if props["RoutePolling"] != "V1_RoutePolling":
            route_polling_val = string_to_boolean(props["RoutePolling"])
            if route_polling_val is None:
                wg_cn_errors.append(f"RoutePolling misconfigured, must be True or False")
            else:
                props["RoutePolling"] = route_polling_val
            logger.info(
                        f"RoutePolling configured to '{props['RoutePolling']}'"
                    )
    else:
        props["RoutePolling"] = LibConfig.Defaults.ROUTE_POLLING
        logger.info(
            f"RoutePolling not configured, can be True or False, setting to '{props['RoutePolling']}'"
        )

    if "RoutePollingInterval" in props:
        interval_val = calc_string_seconds(props["RoutePollingInterval"])
        if interval_val is None:
            props["RoutePollingInterval"] = LibConfig.Defaults.POLLING_INTERVAL
            logger.info(
                "RoutePollingInterval is not specified, using default: %s"
                % LibConfig.Defaults.POLLING_INTERVAL
            )
        elif interval_val < LibConfig.MIN_POLLING_INTERVAL:
            props["RoutePollingInterval"] = LibConfig.MIN_POLLING_INTERVAL
            logger.info(
                "RoutePollingInterval less than minimum value. Using min: %s"
                % LibConfig.MIN_POLLING_INTERVAL
            )
        else:
            props["RoutePollingInterval"] = interval_val
        logger.info(
                    f"RoutePollingInterval configured to '{props['RoutePollingInterval']}' seconds"
                )
    else:
        # Add default if DNE
        props["RoutePollingInterval"] = LibConfig.Defaults.POLLING_INTERVAL
        logger.info(
            f"RoutePollingInterval not configured, setting to '{props['RoutePollingInterval']}' seconds"
        )

    if "HandshakeTimeout" in props:
        timeout_val = calc_string_seconds(props["HandshakeTimeout"])
        if timeout_val is None:
            wg_cn_errors.append(
                "HandshakeTimeout misconfigured, must be integer ending with 's, m, or h'"
            )
        elif timeout_val < LibConfig.MIN_HANDSHAKE_TIMEOUT:
            props["HandshakeTimeout"] = LibConfig.MIN_HANDSHAKE_TIMEOUT
            logger.info(
                "HandshakeTimeout less than minimum value. Using min: %s"
                % LibConfig.MIN_HANDSHAKE_TIMEOUT
            )
        else:
            props["HandshakeTimeout"] = timeout_val
            logger.info(
                        f"HandshakeTimeout configured to: '{props['HandshakeTimeout']}'"
                    )
    else:
        # Add default if DNE.
        props["HandshakeTimeout"] = LibConfig.Defaults.HANDSHAKE_TIMEOUT
        logger.info(
            f"HandshakeTimeout not configured, setting to '{props['HandshakeTimeout']}'"
        )

    if "LocalRoutes" in props and props["LocalRoutes"] != "":
        for route in props["LocalRoutes"].split(","):
            if ":" not in route and not re.match(
                wireguard.CIDR_CHECK_REGEX,
                route.strip(),
            ):
                wg_cn_errors.append(f"Local route not in CIDR format: {route}")
    else:
        props["LocalRoutes"] = ""
        logger.info(f"No local routes specified in LocalRoutes")

   
    if "[" in props["Endpoint"]:
        props["Endpoint"] = props["Endpoint"].replace("[","").replace("]","")

    if "SecondaryEndpoints" not in props:
        props["SecondaryEndpoints"] = ""
        logger.info(f"Secondary endpoints not configured, assuming None")
    elif props["SecondaryEndpoints"] == "":
        logger.info(
            f"No secondary controller endpoints specified in SecondaryEndpoints"
        )
    else:
        endpoint = get_current_endpoint(props)
        if endpoint not in props["SecondaryEndpoints"]:
           props["SecondaryEndpoints"] = f"{endpoint}, {props['SecondaryEndpoints']}"
        logger.info(f"Peered endpoints configured: {props['SecondaryEndpoints']}")

    if 'OpenIdCallbackHost' not in props:
        props["OpenIdCallbackHost"] = "" 
        logger.debug(f"OpenIdCallbackHost not specified, will use default endpoint if needed")
    elif props["OpenIdCallbackHost"] != "":
        logger.info(f"OpenIdCallbackHost configured for OIDC: {props['OpenIdCallbackHost']}")
    else:
        logger.debug("OpenIdCallbackHost not configured")

    props["NoVIPIncluded"] = False
    if 'ControllerVIP' not in props or props["ControllerVIP"] == "":
        props["ControllerVIP"] = ""
        props["NoVIPIncluded"] = True
        logger.info(f"ControllerVIP not specified, controller port 8000 access required")
    
    else:
        logger.info(f"ControllerVIP: {props['ControllerVIP']}, controller port 8000 access not required")

    if "APItimeout" in props and props["APItimeout"] != "":
        api_timeout_val = calc_string_seconds(props["APItimeout"])
        if api_timeout_val is None:
            wg_cn_errors.append(
                "APItimeout misconfigured, must be integer ending with 's, m, or h'"
            )
        else:
            LibConfig.DEFAULT_CONNECT_TIMEOUT = api_timeout_val

    # if validate_endpoints:
    #     # TODO: DONT MERGE, THIS NEEDS TO CHANGE. can we remove?
    #     valid_endpoint, endpoint_errors = get_first_valid_endpoint(env, props,
    #         get_secondary_endpoints_props(props), logger
    #     )
    #     if valid_endpoint:
    #         props = update_props_endpoint(props, valid_endpoint)
    #     if endpoint_errors:
    #         wg_cn_errors = wg_cn_errors + endpoint_errors
    errors = {}
    if wg_native_errors:
        errors[ConnectionTypes.WIREGUARD_NATIVE] = wg_native_errors
    if wg_cn_errors:
        errors[ConnectionTypes.WIREGUARD_COHESIVE] = wg_cn_errors

    logger.debug(f"Config properties validated")
    return errors

def update_openid_callback_host(props, callback_host):
    if 'OpenIdCallbackHost' not in props or not props['OpenIdCallbackHost']:
      props['OpenIdCallbackHost'] = callback_host 

def do_route_polling(props, logger):
    if props["type"] == ConnectionTypes.WIREGUARD_NATIVE:
        return False
    if props.get("TunnelAllTraffic", LibConfig.Defaults.TUNNEL_ALL_TRAFFIC):
        logger.info("TunnelAllTraffic True, no route polling required")
        return False
    elif props.get("RoutePolling", LibConfig.Defaults.ROUTE_POLLING):
        return True
    elif props["RoutePolling"] == "V1_RoutePolling":
        return True
    return False


def tunnel_all_traffic(props):
    if props["type"] == ConnectionTypes.WIREGUARD_NATIVE:
        return False
    return props.get("TunnelAllTraffic", LibConfig.Defaults.TUNNEL_ALL_TRAFFIC)

def verify_vns3_connectivity(env, props, timeout=None, logger=None):
    vns3_endpoint = get_current_endpoint(props)
    vns3_vip = get_current_vip(props, env.logger)
    logger = logger or lib_logger
    logger.debug(f"Pinging controller: {vns3_endpoint}")
    try:
        response = requests.get(
            f"https://{vns3_vip}:8000/api/system/ping",
            timeout=(timeout or LibConfig.DEFAULT_CONNECT_TIMEOUT), verify=False,)
        if response.status_code == 200:
            logger.debug(f"Controller is up via {vns3_vip}")
            return True
        else:
            logger.debug(f"Ping controller failed via {vns3_vip} {response}")
            return False
    except Exception as e:
        logger.debug(f"Failed to ping controller: {vns3_endpoint}:error:%s" % str(e))
        return False


def handle_signal_exit(env, *args, **kwargs):
    sys.exit(0)


def teardown_connection(env, client_props):
    """teardown_connection

    Run teardown steps for VPN connection

    Args:
        env (config.EnvConfig):
        client_props (dict): required properties:
            ActiveInterface: str,
            wg_conf_file: str
            type: str,
            ClientpackId: str,
            Secret: str,
            openid: str,
            refresh_token: str,
            Endpoint: str

    Returns:
        _type_: _description_
    """
    #if wireguard.is_interface_down(env, client_props['ActiveInterface']):
    #    return
    
    logger = env.logger or lib_logger
    client_interface = client_props["ActiveInterface"]
    
    if client_interface == None:
        return

    logger.debug(f"Shutting down Wireguard interface {client_props['ActiveInterface']}")

    if no_vip_included(client_props):
       controller = get_current_endpoint(client_props)
    else:
       controller = get_current_vip(client_props, logger)

    if client_props["type"] == ConnectionTypes.WIREGUARD_COHESIVE:
        stop_route_polling(env, client_props)
        try:
            wg_disconnect(
                controller,
                client_props["ClientpackId"],
                client_props["Secret"],
                logger=logger,
            )
        except Exception as e:
            traceback.print_exc(file=sys.stdout)
            logger.error("Shutdown disconnect error: %s" % str(e))
            logger.debug(traceback.format_exc())

    delete_all_routes(env, client_props)

    try:
        wireguard.wg_quick_down(env, client_interface, client_props["wg_conf_file"])
    except Exception as e:
        traceback.print_exc(file=sys.stdout)
        logger.error(
            "Shutdown interface down error. Please run as admin/sudo. Error: %s"
            % str(e)
        )

    if not wireguard.is_interface_down(env, client_interface):
        logger.error(
            (
                "Failed to down interface %s for connection %s"
                % (client_interface, client_props.get("name", ""))
            ).strip()
        )
        return False

    return True

def disconnect(env, props):
    keys_to_remove = ["pid", "runtime", "token", "openid", "refresh_token"]
    if env.is_macos:
        keys_to_remove.append("interface")

    state_file = env.state_file or props.get("state_file")
    if state_file:
        file_utils.update_data_file(
            state_file,
            {
                "status": "disconnected",
                "last_disconnect": datetime.utcnow().strftime(
                    LibConfig.DATETIME_FORMAT
                ),
            },
            remove_keys=keys_to_remove,
        )
    remove_endpoint_from_host_file(env, props)

def exit_handler(env, props, *args, **kwargs):
    if not wireguard.is_interface_down(env, props['ActiveInterface']):
        env.logger.debug("Stopping connection %s" % props["name"])
        teardown_connection(env, props)
    disconnect(env, props)
    #sys.exit(0)


def string_to_boolean(val):
    if type(val) is bool:
        return val
    if val == "":
        return None
    val_lower = val.lower()
    if not any([val_lower == "true", val_lower == "false"]):
        return None
    return val_lower == "true"


def calc_string_seconds(val, default_unit="s"):
    """calc_string_seconds

    Args:
        val (str): string of form [digits][s,m,h]
        default_unit (str, optional): Defaults to "s".

    Returns:
        None or integer: if None is returned, string is invalid.
    """
    if val == "":
        return None
    elif type(val) in [int, float]:
        return int(val)

    # matches [integer](s,m,h) where s,m,h optional.
    parts_match = re.search(r"([0-9]*)([smh]{0,1})", val)
    if not parts_match:
        return None

    number = parts_match[1]
    matched_unit = parts_match[2]

    if not matched_unit:
        if len(number) == len(val):
            # ie number.isdigit() is True
            unit = default_unit
        else:
            # indicates invalid string
            return None
    else:
        unit = matched_unit

    seconds_per_unit = {"s": 1, "m": 60, "h": 3600}
    if unit not in seconds_per_unit:
        return None

    return int(number) * seconds_per_unit[unit]


def build_client_props(clientpack_file, conn_data):
    props = parse_clientpack(clientpack_file)
    props["name"] = conn_data["name"]
    props["wg_conf_file"] = props["wg_cp_file"]
    props["state_file"] = conn_data.get("state_file")
    props["routes_file"] = conn_data.get("routes_file")
    props["type"] = conn_data.get("type") or ConnectionTypes.DEFAULT
    return props
    
# fetch routes in a separate timer thread    
def route_fetch_timer_function(env, props):
    while 'route_fetch_thread' in props:
        update_routes(
                    env,
                    props,
                    props["routes_file"],
                    props["ControllerVIP"],
                    props["ActiveInterface"],
                    props["ClientpackId"],
                    props["Address"],
                )
        time.sleep(props["RoutePollingInterval"])

def supports_route_listener(props):
    # vns3_version passed from controller
    return "vns3_version" in props and props["RoutePolling"] != "V1_RoutePolling"

def start_route_polling(env, props):
    #stop_route_polling(env, props)
    check_route_polling(env, props)
    if supports_route_listener(props):
        if 'routing_agent' not in props:
            env.logger.debug("Starting routes listener")
            props['routing_agent'] = RoutingAgent(props['ActiveInterface'], logger=env.logger)
            props['routing_agent'].fetch_routes(get_current_vip(props, env.logger))
            props['routing_agent'].start()
        else:
            env.logger.debug("Resetting routes listener")
            props['routing_agent'].stop()
            props['routing_agent'].start()
        
    else:
        if 'route_fetch_thread' not in props:
            env.logger.debug("Starting routes fetcher")
            route_fetch_thread = threading.Thread(target=route_fetch_timer_function, args=(env, props))
            route_fetch_thread.daemon = True #exit thread with main program exit
            props['route_fetch_thread'] = route_fetch_thread
            route_fetch_thread.start()

# when failing over to a peer which may have
# not have support for route listening
def stop_route_polling(env, props): 
    if supports_route_listener(props):
        if 'routing_agent' in props:
            props['routing_agent'].stop()
            props.pop('routing_agent')
    else:
        if 'route_fetch_thread' in props:
            props.pop('route_fetch_thread')


# when failing over to a peer which may have
# not have support for route listening
def check_route_polling(env, props): 
    if supports_route_listener(props):
        if 'route_fetch_thread' in props:
            props.pop('route_fetch_thread')
    else:
        if 'routing_agent' in props:
            props['routing_agent'].stop()
            props.pop('routing_agent')
        

def check_local_gateway_change(env, props):
    while True:
        if props.get('ConnectionStatus') == 'up':
            current_gateway, err = get_default_gateway(env, props)
            if not err:
                if 'local_gateway' not in props:
                    props['local_gateway'] = current_gateway
                elif props['local_gateway'] != '' and current_gateway != props['local_gateway']:
                    env.logger.info(f"Local gateway change detected: {current_gateway}")
                    add_controller_routes(env, props, current_gateway)
                    props['local_gateway'] = current_gateway
        time.sleep(5)

def run_client(env, params, should_thread_stop=None):
    """run_client

    Blocking process. Build WG connection to VNS3.

    Args:
        env (config.EnvConfig)
        params (dict): {
            clientpack: str, path to clientpack file,
            routes: file path, (required),
            state_file: file path, path to write state key:value pairs to
            interface: expected interface to use.
            log_file: file path, library output if None (which defaults to stdout),
            log_level: str, override env log level for file output
        }

    Raises:
        ClientException: str
    """
    routes_file = params.get("routes_file")
    wg_cp_file = params.get("clientpack")

    state_file = params.get("state_file")
    log_file = params.get("log_file")
    params_logger = params.get("logger")
    interface = params.get("interface")
    conn_type = params.get("type") or ConnectionTypes.DEFAULT

    running_in_thread = should_thread_stop is not None
    if not running_in_thread:
        should_thread_stop = lambda: False

    log_level = params.get("log_level", env.get("log_level")).upper()
    if params_logger:
        logger = params_logger
    elif log_file:
        logger = configure_client_logger(
            params["name"], log_level, params.get("log_file"), env.get("log_format")
        )
    else:
        logger = lib_logger

    env.state_file = state_file
    env.logger = logger

    # if not env.is_admin:
    #     error = "Permissions error: must run as administrator"
    #     logger.error(error)
    #     raise ClientException(error)

    if not wg_cp_file:
        error = "Clientpack (conf file) is required for run client"
        logger.error(error)
        raise ClientException(error)

    if not os.path.exists(wg_cp_file):
        error = "Clientpack file not found %s" % wg_cp_file
        logger.error(error)
        raise ClientException(error)

    logger.info("========== Starting Connection ==========")
    logger.info("Client params: [params=%s]" % json_stringify(params))
    logger.info(f"{LibConfig.PROJ_DIR}")
    logger.info(f"{LibConfig.CLI_PATH}")

    urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
    # NOTE: wg_cp_file is a clientpack! wg_conf_file is Wireguard conf file!
    # Question: Should we entirely remove the default and enforce clientpack param
    # is passed along?
    # parse clientpack tokens from file
    props = build_client_props(wg_cp_file, params)
    # validation mutates props
    validation_errors = validate_clientpack(props, logger)
    if validation_errors:
        native_wg_errors = validation_errors.get(ConnectionTypes.WIREGUARD_NATIVE, [])
        wg_cn_errors = validation_errors.get(ConnectionTypes.WIREGUARD_COHESIVE, [])

        # any errors fail for CN type
        if conn_type == ConnectionTypes.WIREGUARD_COHESIVE:
            error = "Problems were found starting up wireguard client: %s" % (
                native_wg_errors + wg_cn_errors
            )
            logger.error(error)
            raise ClientException(error)
        else:
            if len(native_wg_errors) > 0:
                error = (
                    "Problems were found starting up native wireguard: %s"
                    % native_wg_errors
                )
                logger.error(error)
                raise ClientException(error)

    with_route_polling = do_route_polling(props, logger)
    if with_route_polling and not routes_file:
        error = "routes_file param is required if conf is configured for route polling"
        logger.error(error)
        raise ClientException(error)
    props['do_route_polling'] = with_route_polling

    # copy clientpack to wg dir
    initialize(env, props)

    state_updates = {}
    if running_in_thread:
        state_updates["runtime"] = "thread"
        state_updates["pid"] = threading.get_ident()
    else:
        state_updates["runtime"] = "subprocess"
        state_updates["pid"] = os.getpid()

    # Notes about interfaces
    # - Wireguard expects conf file to be named [interface].conf
    # - Mac does not respect interface name but instead creates next available
    #    utun interface (even if conf file is utun5.conf). Linux and Windows
    #    seem to create interfaces by same name as conf file.
    # So for mac basically we have to "find" the interface created for a particular
    # conf file. we do this based on overlay IP (find_ip_interface)

    # if interface not passed explicitly, use clientpack name
    if not interface:
        # use interface parsed from clientpack name
        interface = props["interface"]

    if env.is_macos:
        props["ActiveInterface"] = None
    else:
        props["ActiveInterface"] = interface
        state_updates.update(interface=interface)

    exit_handler_bound = partial(handle_signal_exit, env)
    if not running_in_thread:
        signal_code = signal.SIGBREAK if env.is_windows else signal.SIGTERM
        signal.signal(signal_code, exit_handler_bound)
        atexit.register(exit_handler, env, props)

    file_utils.update_data_file(env.state_file, state_updates)

    default_client_monitor_interval = env.get("conn_monitor_interval")
    login_retry_interval = env.get("login_retry_interval")
    polling_interval = props["RoutePollingInterval"]
    handshake_timeout_interval = props["HandshakeTimeout"]
    props_copy = props.copy()
    props_copy['PrivateKey'] = '<hidden>'
    props_copy['Secret'] = '<hidden>'
    if 'PresharedKey' in props_copy:
        props_copy['PresharedKey'] = '<hidden>'
    logger.debug("Starting main process loop with properties: %s" % props_copy)

    try:
        running_native_only = props["type"] == ConnectionTypes.WIREGUARD_NATIVE
        if running_native_only:
            logger.info("Running native wireguard.")
        else:
            env.logger.debug("Starting gateway monitor")
            check_local_gateway_thread = threading.Thread(target=check_local_gateway_change, args=(env, props))
            check_local_gateway_thread.daemon = True #exit thread with main program exit
            check_local_gateway_thread.start()
            
            # generate wg config file
            update_wg_config(env, props, logger=env.logger)

            logger.debug(
                "Wireguard interface will be checked if no handshake within %s seconds"
                % handshake_timeout_interval
            )

        # Question: should we fail IF there is already a running wg process
        # for the same interface? WG behavior is to down any wg interface with
        # the same name even if its a totally different conf file.
        # So we would have to determine that the current running interface
        # was NOT the one as configured by this clientpack. Perhaps by looking
        # at pass clientpacks Address, and finding IP for the interface.
        # Note: this would not be possible on mac
        if 'ActiveInterface' in props and props['ActiveInterface'] != None:
            wireguard.wg_quick_down(env, props["ActiveInterface"], props["wg_conf_file"])
        
        remove_endpoint_from_host_file(env, props)

        while True:
            #leaving this commented out for future use
            #display_interface_status(
            #    env, props["ActiveInterface"], handshake_timeout_interval
            #)
            if needs_interface_reset(env, props, handshake_timeout_interval):
                teardown_connection(env, props)
                reset_interface(env, props["ActiveInterface"], props, logger=env.logger)

            if props["ConnectionStatus"] != "up":
                time.sleep(login_retry_interval)
            
            else:
                time.sleep(default_client_monitor_interval)

            if should_thread_stop():
                env.logger.info("Thread stop request received. Shutting down.")
                break

        # if it gets here, we are exiting!
        exit_handler(env, props)
    except Exception as e:
        traceback.print_exc(file=sys.stdout)
        env.logger.info(traceback.format_exc())
        env.logger.error("Fatal exception for client: %s" % str(e))
        if log_level == "DEBUG":
            traceback.print_exc(file=sys.stdout)
            env.logger.debug(traceback.format_exc())

        if running_in_thread:
            exit_handler(env, props)
        else:
            remove_endpoint_from_host_file(env, props)
            sys.exit(1)
