import argparse
import logging
import json
import os
import sys
from typing import List, Dict

import cnvpnclient.api as cn_client_api
import cnvpnclient.config as config
from cnvpnclient.logutil import setup_root_logging_stdout
from cnvpnclient.exception import ClientException

ConnectionTypes = config.ConnectionTypes


class ArgumentException(Exception):
    pass


# cli wrapper functions for nicer interfaces with library


def stop_action(env, parsed_args):
    cli_args_raw = parsed_args.get("action_args")
    cli_args = cli_args_raw.split(",") if cli_args_raw else []
    name_kwarg = parsed_args.get("name")
    name_arg = cli_args[0] if len(cli_args) else None

    if name_kwarg and name_arg:
        raise ClientException(
            "Name param and name arg provided. Please provide only 1."
        )

    name = name_kwarg or name_arg
    if name == "all":
        return cn_client_api.stop_all_connections(env=env)

    return cn_client_api.stop_connection(name, env=env)


def start_action(env, parsed_args):
    cli_args_raw = parsed_args.get("action_args")
    cli_args = cli_args_raw.split(",") if cli_args_raw else []
    name_kwarg = parsed_args.get("name")
    name_arg = cli_args[0] if len(cli_args) else None

    if name_kwarg and name_arg:
        raise ClientException(
            "Name param and name arg provided. Please provide only 1."
        )

    kwargs = {"env": env}

    _, passed_kwargs = parse_method_params(parsed_args)
    if passed_kwargs:
        kwargs.update(passed_kwargs)

    name = name_kwarg or name_arg
    if name == "all":
        return cn_client_api.start_all_connections(**kwargs)

    clientpack = parsed_args.get("clientpack")
    if clientpack:
        kwargs["clientpack"] = clientpack

    if name:
        kwargs["name"] = name

    conn_type = parsed_args.get("conn_type")
    if conn_type:
        kwargs["conn_type"] = conn_type

    return cn_client_api.start_connection(**kwargs)


ApiActions = {
    "inspect": {
        "method": cn_client_api.inspect_self,
        "kwargs": ["env"],
    },
    "start": start_action,
    "save": {
        "method": cn_client_api.save_connection,
        "kwargs": ["name", "clientpack", "env", "conn_type"],
    },
    "run": {
        "method": cn_client_api.run_connection,
        "kwargs": ["name", "clientpack", "conn_type", "env"],
    },
    "logs": {
        "method": cn_client_api.get_logs,
        "args": ["name"],
        "kwargs": ["env", "lines", "tail"],
    },
    "set-ownership": {
        "method": cn_client_api.set_file_permissions,
        "args": ["name", "user"],
        "kwargs": ["env"],
    },
    "stop": stop_action,
    "delete": {
        "method": cn_client_api.delete_connection,
        "args": ["name"],
        "kwargs": ["env"],
    },
    "list": {
        "method": cn_client_api.list_connections,
        "kwargs": ["names", "env", "verify_state"],
    },
    "show": {
        "method": cn_client_api.get_connection_data,
        "args": ["name"],
        "kwargs": ["env", "verify_state"],
    },
    "interface-down": {
        "method": cn_client_api.down_interface,
        "args": ["name"],
        "kwargs": ["env"],
    },
}


def should_proceed(msg=""):
    answer = input("\n> Proceed %s? [y/n] \n==> " % msg)
    if answer.lower() == "y":
        return
    raise Exception("Forced stop")


def build_cli_parser():
    examples_str = "Actions:\n"
    examples_str += (
        "# start with create new data files (if new name) and start connection\n"
        "$ cnvpn start [all] -n/--name [connection name] (--type [type str])\n"
        "# save will save connection data files but not start/stop connection\n"
        "$ cnvpn save -cp/--clientpack clientpack.conf -n/--name [connection name] (--type [type str])\n"
        "$ cnvpn stop [all] -n/--name [connection name]\n"
        "$ cnvpn delete -n/--name [connection name]\n"
        "$ cnvpn show -n/--name [connection name]\n"
        "$ cnvpn list [--names [name1] [name2]]\n"
        "$ cnvpn logs -n/--name [name] [-f/--tail] [-l/--lines (integer)]\n"
        "# Review version and env\n"
        "$ cnvpn inspect\n\n"
        "Environment variables:\n"
        "[Required]\n"
        "CNVPN_WG_DIR=/path/to/wireguard/ OR 'local' - Where to store wireguard confs. such as wg installation dir. If local, will store local to data directory.\n"
        "[Optional]\n"
        "CNVPN_DATA_DIR=/my/data - Where to store connection data. Default: [User home]/.cohesive (~/.cohesive) .\n"
        "CNVPN_WG_PATH=/path/to/wg - Path to 'wg' executable. Default: wg (assumes in path)\n"
        "CNVPN_WG_QUICK_PATH=/path/to/wg-quick - Path to 'wg-quick' executable. Default: wg-quick (assumes in path)\n"
        "CNVPN_AUTO_DISCOVER_INTERFACE=(False/True) - calculate next available wg interface. e.g. 100_64_0_1.conf => wg2\n"
        "CNVPN_ALLOW_CUSTOM_INTERFACE=(False/True) - allow custom interface name parsed from clientpack file name. e.g. test.conf => test\n"
        "CNVPN_LOG_LEVEL=(ERROR/WARN/INFO/DEBUG) - default INFO\n"
    )

    examples_str += "\n\nEND of HELP"

    parser = argparse.ArgumentParser(
        description="Command line interface for VNS3 VPN Client",
        formatter_class=argparse.RawTextHelpFormatter,
        prog="cnvpn",
        usage="%(prog)s [-h|--help] [--log-level LOG_LEVEL] [-c/-cp clientpack.conf] [-n name] action",
        epilog=examples_str,
    )

    parser.add_argument(
        "-c", "-cp", "--clientpack", dest="clientpack", help="Clientpack file"
    )
    parser.add_argument("-n", "--name", dest="name", help="Connection name")
    parser.add_argument(
        "-t",
        "--type",
        dest="conn_type",
        help="Connection type. Defaults to %s. Options: %s"
        % (ConnectionTypes.DEFAULT, ConnectionTypes.ALL),
        default=None,
    )
    parser.add_argument(
        "-f",
        "--tail",
        dest="tail",
        action="store_true",
        help="Tail logs for logs func",
        default=None,
    )
    parser.add_argument(
        "-l",
        "--lines",
        dest="lines",
        help="Number of log lines to show",
        default=None,
        type=int,
    )
    parser.add_argument("--log-level", dest="log_level", default=None)
    parser.add_argument("action", help="API object action to execute", type=str)
    parser.add_argument(
        "action_args", help="Pass argument to action (can be csv)", type=str, nargs="?"
    )
    parser.add_argument(
        "--names",
        dest="names",
        nargs="?",
        type=lambda v: v.split(","),
        help="filter by clientpack names",
    )
    parser.add_argument(
        "-p",
        "--param",
        dest="params",
        action="append",
        metavar="P",
        help="A parameter to pass to method call. Name parameters with -p foo=bar.",
    )
    # parser.add_argument(
    #     "--details",
    #     action="store_true",
    #     dest="details",
    #     help="Show full details of connections",
    # )

    return parser


def call_method(view_cls, method_name: str, args: List = [], kwargs: Dict = {}):
    view = view_cls()
    method = getattr(view, method_name)
    resp = method(*args, **kwargs)

    if not resp:
        return
    elif type(resp) in (list, dict):
        print(json.dumps(resp, indent=4, sort_keys=True, default=str))
    else:
        print(resp)


def print_exception(exc, traceback=False):
    exception_str = str(exc)
    if "missing" in exception_str:
        # if "missing" from TypeError
        # then slice string to remove method signature info
        exception_str = exception_str[exception_str.index("missing") :]
    print("\nException: %s\n" % exception_str)
    if traceback:
        raise exc


def parse_method_params(parsed_args: Dict):
    """Parse args dict into args list and kwargs dict

    Arguments:
        args [Dict]

    Returns:
        [tuple] -- (args: List, kwargs: Dict)
    """
    method_args = []
    method_kwargs = {}
    for param in parsed_args.get("params") or []:
        parts = param.split("=")
        parts_len = len(parts)
        if parts_len == 0:
            continue
        elif parts_len == 1:
            method_args.append(parts[0])
        else:
            val = parts[1]
            if "," in val:
                val = val.split(",")
            elif val.lower() in ("true", "false"):
                val = val.lower() == "true"
            method_kwargs[parts[0]] = val
    return method_args, method_kwargs


def run(args=None):
    """
    Command line interface runner
    """
    # this is a little hacky. but it allows for a proxy param for the cli
    # which is used by mac gui currently
    proxy_args = None
    if args:
        proxy_args = sys.argv[1:]
        for a in args:
            proxy_args.remove(a)

    parser = build_cli_parser()
    parsed_args = vars(parser.parse_args(args=args))

    logger = None
    try:
        action = parsed_args["action"]
        if action not in ApiActions:
            print_exception(
                'Unknown action "%s". Valid actions: %s'
                % (action, list(ApiActions.keys()))
            )
            parser.print_help()
            sys.exit(1)

        if action != "run":
            logger = setup_root_logging_stdout(parsed_args.get("log_level"))
            logger.debug("Running cohesive client cli with args: %s" % parsed_args)

        action_config = ApiActions[action]
        if callable(action_config):
            method = action_config
            env = config.read_env(CLI_PROXY_ARGS=proxy_args)
            method_args = (
                env,
                parsed_args,
            )
            method_kwargs = {}
        else:
            passed_args, passed_kwargs = parse_method_params(parsed_args)
            method = action_config.get("method")
            if not method:
                raise RuntimeError(
                    "Invalid actions configured. Contact Cohesive support."
                )

            valid_method_kwargs = action_config.get("kwargs", [])
            valid_method_args = action_config.get("args", [])

            method_kwargs = {}
            if valid_method_kwargs:
                for name in valid_method_kwargs:
                    cli_val = parsed_args.get(name)
                    if cli_val is not None:
                        method_kwargs[name] = cli_val
                        continue
                    cli_val_2 = passed_kwargs.get(name)
                    if cli_val_2 is not None:
                        method_kwargs[name] = cli_val_2
                        continue

                # nifty: if env is valid kwarg for method
                # pass it in
                if "env" in valid_method_kwargs:
                    method_kwargs["env"] = config.read_env(CLI_PROXY_ARGS=proxy_args)

            method_args = []
            cli_args_raw = parsed_args.get("action_args")
            cli_args = cli_args_raw.split(",") if cli_args_raw else []
            if valid_method_args:
                if cli_args:
                    method_args = cli_args

                for name in valid_method_args:
                    cli_val = parsed_args.get(name)
                    if cli_val is not None:
                        method_args.append(cli_val)
            elif cli_args:
                if logger:
                    # should we fail here?
                    logger.error(
                        "Action %s does not accept un-named argument params via CLI. Received: %s"
                        % (action, cli_args)
                    )
                    raise ClientException(
                        "Invalid un-named params provided: %s" % str(cli_args)
                    )

            if passed_args:
                method_args = method_args + passed_args

        response = method(*method_args, **method_kwargs)

        if not response:
            return
        elif type(response) in (list, dict):
            print(json.dumps(response, indent=4, sort_keys=True, default=str))
        else:
            print(response)

        sys.exit(0)
    except ClientException as e:
        print_exception(e, traceback=False)
        sys.exit(1)
    except (AttributeError, TypeError, Exception) as e:
        print_exception(e, traceback=True)
        sys.exit(1)


if __name__ == "__main__":
    """
    Command line interface runner. allow direct call cli.py
    """
    run()
