## Cohesive VPN Client Management API

import threading
import re, requests, os, subprocess, datetime, time, glob, shutil, logging
from urllib3.exceptions import InsecureRequestWarning
from functools import partial

import cnvpnclient.files as file_utils
from cnvpnclient.logutil import configure_client_logger
from cnvpnclient import os_utils, client, wireguard
from cnvpnclient.version import VERSION
from cnvpnclient.exception import ClientException
from cnvpnclient.config import EnvConfig, LibConfig, ConnectionTypes

requests.packages.urllib3.disable_warnings(category=InsecureRequestWarning)
lib_logger = logging.getLogger("cnvpnclient")


def inspect_self(*args, env=None, **kwargs):
    env = env or EnvConfig()
    data = {"version": VERSION, "env": env.config, "username": env.username}
    build_number = env.get("build")
    if build_number:
        data["build"] = build_number

    errors = []

    _, wg_version, wg_errors = wireguard.verify_wg(env)
    if wg_version:
        data["wg_version"] = wg_version

    if wg_errors:
        errors.append(
            "wireguard errors: %s"
            % ".".join(["%s: %s" % wge for wge in wg_errors.items()])
        )

    if errors:
        lib_logger.error(errors)
    return data


def set_file_permissions(name, user, env=None, **kwargs):
    if not name:
        raise ClientException("name is required")

    if type(name) is str:
        env = env or EnvConfig()
        data_file_path = env.conn_data_file(name)
        data = parse_connection_data_file(data_file_path)
        if data is None:
            raise ClientException("Connection %s does not exist" % name)
    else:
        assert (
            type(name) is dict
        ), "first param must be either name (string) or data dict"
        data = name
        data_file_path = data["state_file"]

    routes_file = data.get("routes_file")
    log_file = data.get("log_file")
    conf_file = data.get("conf")
    if env.is_posix:
        os_utils.ensure_owner_posix(data_file_path, user=user, allow_fail=True)
        os_utils.ensure_owner_posix(log_file, user=user, allow_fail=True)
        os_utils.ensure_owner_posix(conf_file, user=user, allow_fail=True)
        if routes_file:
            os_utils.ensure_owner_posix(routes_file, user=user, allow_fail=True)
    else:
        raise ClientException("Permission setting not supported for platform")


def kill_subprocess_connection(env, data, **kwargs):
    # what if data isnt a dict?
    assert (
        type(data) is dict
    ), "killing subprocess requires all connection data as dict. data={}"
    pid = data.get("pid")
    if not pid:
        return False

    # try to kill with standard signal code
    result = os_utils.kill_env_process(env, pid)
    if result is None:
        raise ClientException(
            "Permission error: cannot determine process state. Stopping a client process requires admin permissions."
        )

    # should we wait longer?
    max_tries = kwargs.get("max_tries") or 5
    sleep = kwargs.get("sleep") or 0.5
    lib_logger.debug("Stopping process [connection=%s] [pid=%s]" % (data["name"], pid))
    sigterm_killed = os_utils.wait_for_process_killed(
        env, pid, max_tries=max_tries, sleep_secs=sleep
    )
    if sigterm_killed:
        data = parse_connection_data_file(data["state_file"])
        if "pid" not in data:
            return True
        lib_logger.debug("Client process killed but exit handler not called")
    else:
        lib_logger.debug(
            "Client non-responsive to SIGTERM. Killing %s [pid=%s]"
            % (data["name"], pid)
        )

    interface = data.get("interface")
    if not interface:
        lib_logger.warn(
            "Unexpected state: cannot determine interface: Tried to kill subprocess client connection [name=%s] [pid=%s]"
            % (data["name"], pid)
        )
        #process started up but wg connection failed, don't return, continue to tear down connection
        #return False

    if not sigterm_killed:
        os_utils.kill_env_process(env, pid, force=True)

    return True


def kill_connection(data, env=None):
    """kill_connection

    Given data of connection, kill the connection. This handles
    runtime "type". If connection is running in a thread, we stop that thread,
    otherwise we sigkill the process pid.

    Args:
        data (dict|str): connection data OR str. if str, it will assume name or pid
                         depending on the platform
        env (config.EnvConfig, optional):

    Returns:
        Boolean - true if connection killed.
    """
    env = env or EnvConfig()

    assert data, "data cannot be None"

    is_dict = type(data) is dict

    # maybe this is being a bit too nice accepting non-dicts
    if not is_dict:
        # this is assuming data is a pid. Should we force kill in this case?
        return os_utils.kill_env_process(env, data)
    else:
        # assume Subprocess for backwards compat
        runtime = data.get("runtime") or LibConfig.SUBPROCESS
        pid = data.get("pid")
        resp = False
        if runtime == LibConfig.THREAD:
            thread = data.get("thread")
            if thread:
                resp = thread.stop()
            elif pid:
                resp = ClientThread.stop_by_id(pid)
            
        elif pid:
            resp = kill_subprocess_connection(env, data)
        
    # force a disconnect
    client_props = client.build_client_props(data["conf"], data)
    client.validate_clientpack(client_props, lib_logger, validate_endpoints=False)
    file_logger = configure_client_logger(
        data["name"],
        data["log_level"].upper() if "log_level" in data else env.get("log_level").upper(),
        data.get("log_file"),
        env.get("log_format"),
    )

    if file_logger:
        env.logger = file_logger

    client_props["ActiveInterface"] = data.get("interface")
    client.exit_handler(env, client_props)
    return resp


def verify_connection_state(data, interface_details=None, env=None, **kwargs):
    """verify_connection_state

    Here we check the connection data state for its "truthfulness". Basically
    we are checking to see that if status is connected then
    the interface is up and the client process (a subprocess or a thread) is running.
    If the connection is native then only verify interface is up

    Args:
        data (_type_): _description_
        env (_type_, optional): _description_. Defaults to None.

    Returns:
        tuple[dict, str]: (state updates, message string)
    """
    env = env or EnvConfig()
    logger = env.logger or lib_logger
    pid = data.get("pid")
    cur_status = data.get("status")
    conn_type = data["type"]
    conf = data["conf"]
    # assume already fetched. this is only a state checking func

    # state to check.
    process_running = False
    interface_is_up = bool(interface_details)
    interface = data.get("interface")

    if pid:
        runtime = data.get("runtime") or LibConfig.SUBPROCESS
        if runtime == LibConfig.THREAD:
            thread = ClientThread.find_thread(pid)
            process_running = bool(thread and thread.is_alive())
        else:
            logger.debug("Verifying pid %s exists" % pid)
            process_running = os_utils.process_exists(env, pid)

    # None is "i dont know" and is likely a permissions error, meaning process probably exists
    process_is_not_running = process_running is False
    process_might_be_running = not process_is_not_running
    # implications:
    # if Cohesive WG: connected    <=> process is running & interface is up
    #                 disconnected <=> process not running & interface down
    # if Native WG:   connected    <=> interface is up
    #                 disconnected <=> interface is down
    # Question? Should we down the interface if we dont think it should be up?

    state_updates = {}
    message = None
    if conn_type == ConnectionTypes.WIREGUARD_NATIVE:
        if cur_status == "connected" and not interface_is_up:
            state_updates.update(status="disconnected")
            message = (
                "Status connected but interface not up. Status should be disconnected."
            )
        elif cur_status == "disconnected" and interface_is_up:
            state_updates.update(status="connected")
            message = (
                "Status disconnected but interface is up. Status should be connected."
            )
    elif conn_type == ConnectionTypes.WIREGUARD_COHESIVE:
        # could be an orphaned process... should we look for it?
        if process_is_not_running:
            if cur_status in ("connected", "connecting"):
                state_updates.update(status="disconnected")
                message = "Status connected/connecting but client is not running"

            if interface_is_up:
                # should we down this? probably
                wireguard.wg_quick_down(env, interface=interface, conf_file_path=conf)
        elif process_running is None:
            message = (
                "Process %s might be running but os pid query not permitted. Invalid permissions."
                % pid
            )

    if process_is_not_running:
        state_updates["remove_keys"] = ["pid", "runtime"]
        if env.is_macos:
            state_updates["remove_keys"].append("interface")

    # dont allow any state updates if we cant be certain the process is not running
    if process_might_be_running:
        state_updates = {}
    return state_updates, message


def list_connections(names=None, name_only=False, verify_state=True, env=None):
    env = env or EnvConfig()
    conn_file_glob = os.path.join(
        env.get("data_dir"), "%s.*" % LibConfig.CONN_DATA_FILE_PREFIX
    )
    connections = []

    for filename in glob.glob(conn_file_glob):
        connection_name = os.path.basename(filename).replace(
            "%s." % LibConfig.CONN_DATA_FILE_PREFIX, ""
        )

        if names and connection_name not in names:
            continue

        if name_only:
            connections.append(connection_name)
        elif verify_state:
            connections.append(get_connection_data(connection_name, env=env))
        else:
            connections.append(parse_connection_data_file(filename))
    return connections


def read_interfaces(conf_dir):
    return [conf_name.split(".")[0] for conf_name in os.listdir(conf_dir)]


def wg_conf_path(interface, env):
    interface = interface.split(".")[0]  # just in case
    interface_conf = "%s.conf" % interface
    return os.path.join(env.get("wg_dir"), interface_conf)


def get_next_interface(conf_dir=None, interfaces_in_use=None):
    if interfaces_in_use:
        taken_interfaces = interfaces_in_use
    else:
        taken_interfaces = read_interfaces(conf_dir)

    taken_interface_numbers = []
    for i in taken_interfaces:
        if re.match(wireguard.CONF_REGEX, i):
            taken_interface_numbers.append(int(i.replace("wg", "")))
        else:
            continue

    candidate = 0
    next_interface_num = None
    while next_interface_num is None:
        if candidate not in taken_interface_numbers:
            next_interface_num = candidate
        else:
            candidate = candidate + 1

    return "wg%d" % next_interface_num


def parse_connection_data_file(data_file_path, read_sub_files=False):
    data = file_utils.parse_data_file(data_file_path)
    if data is None:
        return None

    if "type" not in data:
        data["type"] = ConnectionTypes.DEFAULT

    data["state_file"] = data_file_path
    if read_sub_files:
        # TODO add read log file?
        # log_file = data.get("log_file")
        # if log_file:
        #     pass

        routes_file = data.get("routes_file")
        if routes_file and os.path.exists(routes_file):
            try:
                # probably wont get here as parse_data_file will likely fail first
                data["routes"] = open(routes_file, "r").readlines()
            except PermissionError:
                pass

    return data


def get_connection_data(name, env=None, verify_state=True):
    logger = env.logger or lib_logger
    if not name:
        raise ClientException("name is required")

    env = env or EnvConfig()
    data_file_path = env.conn_data_file(name)
    data = parse_connection_data_file(data_file_path, read_sub_files=True)
    if data is None:
        raise ClientException("Connection %s does not exist" % name)

    if not verify_state:
        return data

    interface = data.get("interface")
    interface_details, interface_error = None, "no interface found"
    if interface:
        interface_details, interface_error = wireguard.get_interface_details(
            env, env.get("wg_path"), interface
        )

    # we know that the interface IS up if we get an invalid permissions error as otherwise
    # we get a 'unable to access interface' error
    interface_exists = (interface_details is not None) or (
        interface_error and "invalid permissions" in interface_error.lower()
    )

    state_updates, message = verify_connection_state(
        data, interface_details=interface_exists, env=env
    )
    if state_updates:
        if state_updates.get("status") == "disconnected":
            state_updates["remove_keys"] = ["pid", "runtime"]

        data = put_connection_data_file(data_file_path, name, **state_updates)
        logger = env.logger or lib_logger
        logger.debug(
            "Connection %s state out of date. Updating. Reason: %s. updates: %s" % (name, message, state_updates)
        )

    if message:
        data.update(message=message)

    if interface_error and data["status"] != "disconnected":
        data["interface_details"] = {"error": interface_error}
    elif interface_details:
        data["interface_details"] = interface_details

    return data


def get_logs(name, lines=None, tail=False, env=None):
    if not name:
        raise ClientException("name is required")

    env = env or EnvConfig()
    data_file_path = env.conn_data_file(name)
    data = parse_connection_data_file(data_file_path, read_sub_files=True)
    if not data:
        raise ClientException("%s does not exist" % name)

    log_file = data["log_file"]
    if tail:
        if env.is_windows:
            command = [
                "powershell",
                'Get-Content "%s" -Tail %s -Wait' % (log_file, (lines or 50)),
            ]
        else:
            command = ["tail"]
            if lines:
                command.append("-n")
                command.append(str(lines))
            command.append("-F")
            command.append(log_file)

        # will block
        subprocess.call(command)
    else:
        lines = lines or 25
        log_lines_generator = os_utils.reverse_readline(log_file)
        shown = 0
        try:
            while shown < lines:
                print(next(log_lines_generator))
                shown = shown + 1
        except StopIteration:
            pass


def determine_clientpack_path(clientpack, env):
    # [Part 1] Clientpack selection logic
    # 1 Assume F is path to file. Check if file exists. if exists, use F => [Part 2]
    # 2 Look for F in WG_DIR.
    #   F must be valid wgX interface OR ALLOW_CUSTOM_INTERFACE is selected.
    #   if F exists in WG_DIR then
    #       if F in use by another connection
    #           raise exception
    #       else use F => [Part 2]
    if os.path.exists(clientpack):
        return clientpack
    elif clientpack.startswith(".") or clientpack.startswith("/"):
        # if param starts with . or / we assume its a path and so fail if does not exist
        raise ClientException("Client conf %s does not exist." % clientpack)
    else:
        file_name = os.path.basename(clientpack)
        interface_from_file_name = file_name.split(".")[0]
        wg_dir_clientpack_path = os.path.join(
            env.get("wg_dir"), "%s.conf" % interface_from_file_name
        )
        wg_dir_clientpack_exists = os.path.exists(wg_dir_clientpack_path)

        if not wg_dir_clientpack_exists:
            raise ClientException(
                "Client conf %s does not exist. Searched for conf in . and WG_DIR (%s)"
                % (clientpack, env.get("wg_dir"))
            )

        has_valid_wg_interface = re.match(
            wireguard.CONF_REGEX, interface_from_file_name
        )
        if not has_valid_wg_interface and not env.get("allow_custom_interface"):
            raise ClientException(
                "Invalid interface. Clientpack found in %s but not a valid wg interface. Please set CNVPN_ALLOW_CUSTOM_INTERFACE=True."
                % env.get("wg_dir")
            )
        return wg_dir_clientpack_path


def determine_interface_for_new(clientpack_file, env, candidate=None):
    # interface selection logic
    # 1. if ALLOW_CUSTOM_INTERFACE=True or file name is valid wg interface
    #        interface = file base name
    #        check if that interface is taken
    #        if it is taken
    #           if NOT AUTO_DISCOVER_INTERFACE=true, fail
    #           else calculate next interface
    # 2. if AUTO_DISCOVER_INTERFACE=True, calculate next interface
    # 3. if it gets here, raise failed to select interface.
    file_name = os.path.basename(clientpack_file)
    interface_candidate = candidate or file_name.split(".")[0]
    has_valid_wg_interface = re.match(wireguard.CONF_REGEX, interface_candidate)

    auto_discover_interface = env.get("auto_discover_interface")
    allow_custom_interface = env.get("allow_custom_interface")

    _invalid_exception = (
        "Invalid interface %s. Please give the clientpack a valid, available "
        "interface name. e.g. wg0.conf. You can use env configuration "
        "AUTO_DISCOVER_INTERFACE=True or ALLOW_CUSTOM_INTERFACE=True to "
        "calculate next available interface or use any"
        "clientpack name as interface, respectively."
    )
    invalid_exception = _invalid_exception % interface_candidate

    existing_interfaces = read_interfaces(env.get("wg_dir"))
    interface_conf_exists = interface_candidate in existing_interfaces
    if not interface_conf_exists:
        if has_valid_wg_interface or allow_custom_interface:
            return interface_candidate
        elif auto_discover_interface:
            return get_next_interface(interfaces_in_use=existing_interfaces)
        else:
            raise ClientException(invalid_exception)
    else:  # interface candidate conf already exists in wg_dir
        if clientpack_file == wg_conf_path(interface_candidate, env):
            # this works because this func should only be called for saving _new_ connections
            return interface_candidate
        elif auto_discover_interface:
            return get_next_interface(interfaces_in_use=existing_interfaces)
        else:
            raise ClientException(invalid_exception)


def put_connection_data_file(
    data_file_path,
    name,
    clientpack=None,
    interface=None,
    pid=None,
    env=None,
    remove_keys=None,
    **data_fields
):
    # these can be overridden by data_fields
    prev_data = parse_connection_data_file(data_file_path)

    created = prev_data is None
    _data = {"name": name}

    if prev_data:
        prev_clientpack = prev_data.get("conf")
        if clientpack and clientpack != prev_clientpack:
            if os.path.exists(prev_clientpack):
                os.remove(prev_clientpack)
        _data = prev_data

        if remove_keys:
            for key in remove_keys:
                _data.pop(key, None)
    else:
        if not env:
            raise ClientException("env required for new connection")

        # If this is a new connection data file. Set some defaults
        _data["log_file"] = env.conn_log_file(name)
        _data["routes_file"] = env.conn_routes_file(name)
        _data["created"] = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        _data["state_file"] = data_file_path
        _data["status"] = "disconnected"

    if clientpack:
        if not env:
            # make interface explicitly required.
            raise ClientException(
                "env and interface arguments are required if clientpack is passed"
            )

        if not interface and not (prev_data and "interface" in prev_data):
            raise ClientException(
                "interface not passed and not in connection saved data. Cant save new clientpack."
            )

        _data["interface"] = interface or prev_data["interface"]

        wg_dir_conf_file_path = os.path.join(
            env.get("wg_dir"), "%s.conf" % _data["interface"]
        )
        if clientpack != wg_dir_conf_file_path:
            lib_logger.debug(
                "Copying clientpack to WG_DIR (%s)" % wg_dir_conf_file_path
            )
            shutil.copy(clientpack, wg_dir_conf_file_path)

        _data["conf"] = wg_dir_conf_file_path

        try:
            clientpack_raw = open(clientpack, "r").read()
        except PermissionError:
            raise ClientException(
                "Cannot read clientpack file %s. Invalid permissions." % clientpack
            )
        address_match = re.search(r"Address\ *\=\ *([0-9\.\/]*)", clientpack_raw)
        if address_match:
            # this should always match...
            address = address_match[1].split("/")[0]
            _data["overlay_ip"] = address

        endpoint_match = re.search(
            r"Endpoint\ *\=\ *([a-zA-Z0-9\-\.\/\:]*)", clientpack_raw
        )
        if endpoint_match:
            # this should always match...
            endpoint = endpoint_match[1].split(":")[0]
            _data["endpoint"] = endpoint

    if pid:
        _data["pid"] = pid
    elif pid == 0:
        # pid=0 means delete pid. None means don't overwrite pid
        _data.pop("pid", None)

    for key, value in data_fields.items():
        key_lower = key.lower()
        _data[key_lower] = value

    file_utils.put_data_file(data_file_path, _data)

    _data["operation"] = "create" if created else "update"
    return _data


def run_connection(
    name=None, clientpack=None, conn_type=None, env=None, should_thread_stop=None
):
    """
    run_connection - blocking call to run the Cohesive VPN Client

    Possible calls:
        name=None, clientpack=None - invalid
        name=myconnection, clientpack=None - only valid if state file for myconnection
            exists. start connection writes this state file. We pull clientpack conf
            file path from that state file.
        name=None, clientpack=clientpack file path - creates time-based connection name
            and tries to run connection with clientpack file provided
        name=myconnection, clientpack=file - this will use config in the myconnection
            state file and use the clientpack file passed instead of conf saved in
            connection state file
    """
    if not any([name, clientpack]):
        raise ClientException("name or clientpack are required to run connection")

    env = env or EnvConfig()

    conn_data_file, conn_data = None, None
    if name:
        conn_data_file = env.conn_data_file(name)
        conn_data = parse_connection_data_file(conn_data_file)
        if not conn_data and not clientpack:
            raise ClientException("connection name=%s does not exist" % name)
    else:
        name = ".".join(os.path.basename(clientpack).split(".")[:-1])
        conn_data_file = env.conn_data_file(name)

    # Allow just passing a clientpack and name. But typically
    # start_connection will be used.
    if conn_data:
        conn_log_file = conn_data["log_file"]
        conn_routes_file = conn_data["routes_file"]
        if conn_type is None:
            conn_type = conn_data.get("type")
        if clientpack:
            # if clientpack is passed. automatically use that
            # for interface. Might fail if, for instance, wg0.conf
            # is passed but that is currently taken
            interface = os.path.basename(clientpack).split(".")[0]
        else:
            clientpack = conn_data["conf"]
            interface = conn_data.get("interface")

            if not interface:
                interface = os.path.basename(clientpack).split(".")[0]
    else:
        if not clientpack:
            raise ClientException(
                "clientpack is required for unnamed run_client_connect"
            )

        conn_log_file = env.conn_log_file(name)
        # will try to use interface in clientpack. But could fail
        # if interface is already taken. start connection handles
        # this for you. call run_client_connect directly just tries
        interface = os.path.basename(clientpack).split(".")[0]
        # routes file is required
        conn_routes_file = env.conn_routes_file(name)

    # future: might remove this default assumption.
    if conn_type is None:
        conn_type = ConnectionTypes.DEFAULT

    lib_logger.debug(
        "Running client connection name=%s clientpack=%s log_file=%s type=%s"
        % (name, clientpack, conn_log_file, conn_type)
    )

    log_level = env.get("log_level")
    if "log_level" in conn_data:
        log_level = conn_data["log_level"]
    client_file_logger = configure_client_logger(
        name, log_level.upper(), conn_log_file, env.get("log_format")
    )
    client_file_logger.debug("logging level: %s", log_level)

    valid_wg, _, wg_errors = wireguard.verify_wg(env)
    if not valid_wg:
        wg_error_str = ".".join(["%s: %s" % wge for wge in wg_errors.items()])
        error = "wireguard installation invalid: %s" % wg_error_str
        client_file_logger.error(error)
        raise ClientException(error)

    client.run_client(
        env,
        {
            "name": name,
            "state_file": conn_data_file,
            "logger": client_file_logger,
            "routes_file": conn_routes_file,
            "interface": interface,
            "clientpack": clientpack,
            "type": conn_type,
        },
        should_thread_stop=should_thread_stop,
    )


def save_connection(name=None, clientpack=None, env=None, conn_type=None, **kwargs):
    env = env or EnvConfig()

    if not name and not clientpack:
        raise ClientException(
            "save connection requires 'name' and/or 'clientpack' file path."
        )

    is_new = True
    connection_id_args = ()
    save_data = {}
    save_data.update(kwargs)
    save_data["env"] = env
    save_data.pop("interface", None)  # interface is protected. cant pass.
    prev_data = None

    if conn_type is not None and conn_type not in ConnectionTypes.ALL:
        raise ClientException(
            "Invalid connection type. Valid types are %s. Default: %s"
            % (ConnectionTypes.ALL, ConnectionTypes.DEFAULT)
        )

    if name:
        data_file_path = env.conn_data_file(name)
        is_new = not os.path.exists(data_file_path)
        connection_id_args = (data_file_path, name)
        if is_new:
            conn_type = conn_type or ConnectionTypes.DEFAULT
        else:
            prev_data = parse_connection_data_file(data_file_path)

    if is_new and not clientpack:
        raise ClientException(
            "%s does not exist. A clientpack is required for new connections." % name
        )
    elif not is_new and "pid" in prev_data:
        raise ClientException(
            "%s connection is running [status=%s] [pid=%s]. Cannot update connection state while running. Please stop before updating."
            % (name, prev_data["status"], prev_data["pid"])
        )

    if clientpack:
        clientpack_path = determine_clientpack_path(clientpack, env)
        save_data["clientpack"] = clientpack_path

        if name:  # we know not new
            for client in list_connections(verify_state=False):
                if client["conf"] == clientpack_path and client["name"] != name:
                    raise ClientException(
                        "Client with conf %s already exists with name %s."
                        % (clientpack_path, client["name"])
                    )
        else:
            # if passing clientpack and no name, must be a new connection
            name = ".".join(os.path.basename(clientpack_path).split(".")[:-1])
            data_file_path = env.conn_data_file(name)
            connection_id_args = (data_file_path, name)
            conn_type = conn_type or ConnectionTypes.DEFAULT
            if os.path.exists(data_file_path):
                raise ClientException(
                    "Connection with name %s already exists. Please provide a name or rename the clientpack file. To start the connection %s, pass the name only (via --name)."
                    "You can overwrite %s's clientpack by passing the name explicitly via --name along with the clientpack (--clientpack)."
                    % (name, name, name)
                )

        if is_new:
            client_interface = determine_interface_for_new(clientpack_path, env)
            save_data["interface"] = client_interface
        else:
            # not new but new clientpack passed.
            # Use already selected "interface". If linux/windows, it will be consistent
            # If mac, interface changes so we simply rely on conf file name
            if env.is_macos:
                save_data["interface"] = os.path.basename(prev_data["conf"]).split(".")[
                    0
                ]
            else:
                save_data["interface"] = prev_data["interface"]

    if conn_type is not None:
        save_data["type"] = conn_type

    final_data = put_connection_data_file(*connection_id_args, **save_data)

    if env.is_posix and env.username != "root":
        logger = env.logger or lib_logger
        logger.debug("Updating state files owner to %s if possible" % env.username)
        set_file_permissions(final_data, user=env.username, env=env)

    return final_data


def down_interface(name, env=None):
    if not name:
        raise ClientException("name is required to down interface")

    env = env or EnvConfig()
    data_file_path = env.conn_data_file(name)
    conn_data = parse_connection_data_file(data_file_path)
    if not conn_data:
        lib_logger.debug("No connection data for %s" % name)
        return

    interface = conn_data.get("interface")
    if not interface:
        lib_logger.debug("No interface for connection %s" % name)
        return

    if wireguard.is_interface_down(env, interface):
        lib_logger.debug(
            "Connection %s interface (%s) already down" % (name, interface)
        )
        return

    conf = conn_data.get("conf")
    if not os.path.exists(conf):
        lib_logger.debug("Connection %s conf (%s) does not exist" % (name, conf))
        return

    wireguard.wg_quick_down(
        env, interface=conn_data.get("interface"), conf_file_path=conf
    )
    if env.is_macos:
        conn_data.pop("interface", None)
        file_utils.put_data_file(data_file_path, conn_data)

    return True


def start_connection(
    name=None, clientpack=None, conn_type=None, env=None, thread=False, verify_up=True
):
    env = env or EnvConfig()
    if name:
        data_file_path = env.conn_data_file(name)
        is_new = not os.path.exists(data_file_path)
    else:
        is_new = True

    if not env.is_admin:
        raise ClientException("Start must be run with admin/sudo permissions")

    if is_new and clientpack is None:
        raise ClientException(
            "Invalid args. clientpack and/or name is required to create new connection or start an existing connection."
        )

    if is_new:
        if not env or not env.get("wg_dir"):
            raise ClientException(
                "WG_DIR is required when creating a new connection. Please set env variable CNVPN_WG_DIR=/path/to/wireguard-conf-location/ "
                'You can also use "local" to store wireguard client confs in cli state directory ./data.'
            )

        conn_data = save_connection(
            name=name, clientpack=clientpack, conn_type=conn_type, env=env
        )
    else:
        conn_data = parse_connection_data_file(data_file_path)

    name = conn_data["name"]
    data_file_path = env.conn_data_file(name)
    state_updates = {}

    was_running = False
    if "pid" in conn_data:
        # Currently restart if it was running
        was_running = kill_connection(conn_data, env=env)
        if was_running is True:
            lib_logger.debug("Connection was already running. Restarting.")
        elif was_running is None:
            lib_logger.debug(
                "Connection is likely running but current user permissions does not permit querying process with pid %s"
                % conn_data["pid"]
            )
            raise ClientException(
                "Connection already running. Please review process with pid %s and stop/start with proper user permissions (likely running as admin/root)."
                % conn_data["pid"]
            )

        state_updates["remove_keys"] = ["pid"]
        conn_data.pop("pid", None)

    # if a new clientpack or conn_type was provided, need to save that to state
    # Note: this has to happen after we try to kill connection if exists as cant
    # save with proc running
    if clientpack or conn_type:
        conn_data = save_connection(
            name=name, clientpack=clientpack, conn_type=conn_type, env=env
        )

    if conn_data["status"] == "connected" and not was_running:
        state_updates["data"] = {"status": "disconnected"}

    if state_updates:
        file_utils.update_data_file(data_file_path, **state_updates)

    log_msg = "Starting%s client connection %s" % (" new" if is_new else "", name)
    lib_logger.debug(log_msg)
    cli_proxy_args = env.get("cli_proxy_args") or []
    start_command = (
        [env.get("cli_path")] + cli_proxy_args + ["run", "--name", "'%s'" % name]
    )
    py_path = env.get("py_path")
    if py_path:
        if " " in start_command[0]:
            # should really only be for windows.
            start_command[0] = "'%s'" % start_command[0]
        start_command = [py_path] + start_command
        if env.is_windows and py_path.startswith("&"):
            start_command = ["powershell"] + start_command

    client_thread, process = None, None
    if thread:
        client_thread = ClientThread(name)
        client_thread.start()
        check_client = lambda: not client_thread.is_alive()
    else:
        process = subprocess.Popen(
            start_command, stderr=subprocess.DEVNULL, stdout=subprocess.DEVNULL
        )
        check_client = process.poll

    if verify_up:
        checks = 0
        check_threshold = 8
        client_exit = False
        while checks <= check_threshold:
            lib_logger.debug(
                "Process still up. Check=%s. Checks left=%s."
                % (checks, (check_threshold - checks))
            )
            client_exit = check_client()
            if client_exit:
                break
            checks = checks + 1
            time.sleep(0.5)

        if client_exit:
            error = (
                "Failed to start client connection. See log file for more details: %s"
                % conn_data["log_file"]
            )
            lib_logger.error(error)
            raise ClientException(error)

    latest_conn_data = parse_connection_data_file(data_file_path)
    process_updates = {"status": "connecting"}
    if process:
        process_updates.update(pid=process.pid)
    else:
        process_updates.update(thread=client_thread)

    latest_conn_data.update(process_updates)
    return latest_conn_data


def start_all_connections(env=None, reconnect_only=False, as_threads=False):
    env = env or EnvConfig()
    retval = []
    for connection in list_connections(verify_state=False):
        reconnects_config_val = connection.get("reconnect")
        reconnect_configured = (
            reconnects_config_val.lower() == "true" if reconnects_config_val else False
        )
        if reconnect_only and not reconnect_configured:
            continue
        try:
            retval.append(
                start_connection(connection["name"], env=env, thread=as_threads)
            )
        except ClientException as e:
            lib_logger.error(
                "Failed to start connection %s: %s" % (connection["name"], str(e))
            )
    return retval


def stop_connection(name, env=None):
    if not name:
        raise ClientException("name is required to stop client connection")

    env = env or EnvConfig()
    data_file_path = env.conn_data_file(name)
    conn_data = parse_connection_data_file(data_file_path)
    if not conn_data:
        raise ClientException("Connection %s does not exist" % name)

    if not env.is_admin:
        raise ClientException("Stop must be run with admin/sudo permissions")

    lib_logger.debug("Stopping client connection %s" % name)
    killed = kill_connection(conn_data, env)

    if killed is None:
        raise ClientException(
            "Could not determine connection process state. Are you running with correct permissions?"
        )

    _latest_data = parse_connection_data_file(data_file_path)
    if env.is_macos and "interface" in _latest_data:
        _latest_data = file_utils.remove_key(data_file_path, "interface")

    return dict(_latest_data, action=("stopped" if killed else "noop"))


def stop_all_connections(env=None):
    env = env or EnvConfig()
    retval = []
    for connection in list_connections(verify_state=False):
        try:
            retval.append(stop_connection(connection["name"], env=env))
        except ClientException as e:
            lib_logger.error(
                "Failed to stop connection %s: %s" % (connection["name"], str(e))
            )
    return retval


def delete_connection(name, env=None):
    if not name:
        raise ClientException("name required for delete connection")

    env = env or EnvConfig()
    response = stop_connection(name, env=env)
    if response["action"] == "stopped":
        # This is unfortunate but was seeing behavior where client.exit_handler
        # is being called after we removed the routes file. Maybe there is a better
        # way to handle this to gaurantee no errors.
        time.sleep(2)

    data_file_path = env.conn_data_file(name)
    conn_data = parse_connection_data_file(data_file_path)
    if not conn_data:
        return None

    log_file = conn_data.get("log_file")
    routes_file = conn_data.get("routes_file")
    interface_file = conn_data.get("interface_file")
    conf_file = conn_data.get("conf")
    logging.shutdown() #release file locks for windows
    if log_file:
        file_utils.remove_file(log_file)
    if conf_file:
        file_utils.remove_file(conf_file)
    if interface_file:
        file_utils.remove_file(interface_file)
    if routes_file:
        file_utils.remove_file(routes_file)

    file_utils.remove_file(data_file_path)
    return conn_data


class ClientThread(threading.Thread):
    """Client as a thread. This is useful if you are using this library in python vs
    as a CLI. Note, if you are using this, you are responsible for stopping threads
    correctly!! Should call stop()

    client = ClientThread(env, params)
    client.start()
    # stored in application
    # at some later time:
    client.stop()

    Args:
        env (config.EnvConfig):
        params (dict)
    """

    def __init__(self, name):
        super().__init__(daemon=True)
        self.name = name
        self._stop_event = threading.Event()

    def run(self):
        def should_stop():
            return self._stop_event.is_set()

        return run_connection(self.name, should_thread_stop=should_stop)

    def stop(self):
        if not self._stop_event.is_set():
            self._stop_event.set()
            return True
        return False

    @staticmethod
    def validate_ident(ident):
        if type(ident) is str:
            if not ident.isdigit():
                lib_logger.debug("Invalid thread identifier %s" % ident)
                return None
        try:
            return int(ident)
        except ValueError:
            return None

    @classmethod
    def find_thread(cls, ident):
        _ident = cls.validate_ident(ident)
        if not _ident:
            return None

        return threading._active.get(_ident)

    @classmethod
    def stop_by_id(cls, ident):
        _ident = cls.validate_ident(ident)
        if not _ident:
            return False

        thread = cls.find_thread(_ident)
        if not thread:
            lib_logger.debug("Failed to located thread for stop %s" % _ident)
            return False

        if not thread.is_alive():
            return False

        thread.stop()
        return True


class Api(object):
    def __init__(self, env) -> None:
        self.env = env

        # Api methods!
        self.list_connections = partial(list_connections, env=self.env)
        self.get_connection = partial(get_connection_data, env=self.env)
        self.run_connection = partial(run_connection, env=self.env)
        self.start_connection = partial(start_connection, env=self.env)
        self.stop_connection = partial(stop_connection, env=self.env)
        self.delete_connection = partial(delete_connection, env=self.env)
        self.save_connection = partial(save_connection, env=self.env)
        self.inspect = partial(inspect_self, env=self.env)
        self.start_all_connections = partial(start_all_connections, env=self.env)
        self.stop_all_connections = partial(stop_all_connections, env=self.env)
        self.set_file_permissions = partial(set_file_permissions, env=self.env)

        # helpers
        self.get_next_interface = partial(get_next_interface, env.get("wg_dir"))
        self.parse_clientpack = client.parse_clientpack
        self.validate_clientpack = client.validate_clientpack
        self.find_connection_thread = ClientThread.find_thread
